package ru.yandex.direct.hourglass.ydb.storage;

import java.time.Duration;
import java.time.Instant;
import java.util.Collection;

import ru.yandex.direct.hourglass.InstanceId;
import ru.yandex.direct.hourglass.TaskProcessingResult;
import ru.yandex.direct.hourglass.storage.JobStatus;
import ru.yandex.direct.hourglass.storage.PrimaryId;
import ru.yandex.direct.hourglass.storage.TaskId;
import ru.yandex.direct.hourglass.storage.Update;
import ru.yandex.direct.ydb.builder.predicate.Predicate;
import ru.yandex.direct.ydb.builder.querybuilder.UpdateBuilder;
import ru.yandex.direct.ydb.column.Column;

import static ru.yandex.direct.hourglass.ydb.storage.ScheduledTasksStatus.DELETED;
import static ru.yandex.direct.hourglass.ydb.storage.ScheduledTasksStatus.NEW;
import static ru.yandex.direct.hourglass.ydb.storage.ScheduledTasksStatus.PAUSED;
import static ru.yandex.direct.hourglass.ydb.storage.ScheduledTasksStatus.RUNNING;
import static ru.yandex.direct.hourglass.ydb.storage.Tables.SCHEDULED_TASKS;
import static ru.yandex.direct.ydb.builder.querybuilder.UpdateBuilder.set;

public class YdbUpdateImpl implements Update {
    private final YdbStorageImpl storage;
    private final YdbFindImpl find;
    private final InstanceId instanceId;
    private JobStatus setJobStatus = null;
    private JobStatus whereJobStatus = null;
    private Boolean needReschedule = null;
    private UpdateBuilder.SetStatement setStatement;

    public YdbUpdateImpl(YdbStorageImpl storage, String version, InstanceId instanceId,
                         Duration taskHeartbeatExpiration) {
        this.storage = storage;
        this.find = new YdbFindImpl(storage, version, instanceId, taskHeartbeatExpiration);
        this.instanceId = instanceId;
    }

    @Override
    public Update wherePrimaryIdIn(Collection<PrimaryId> primaryIds) {
        find.wherePrimaryIdIn(primaryIds, false);
        return this;
    }

    YdbFindImpl getFindCondition() {
        return find;
    }

    Predicate getPredicate() {
        return find.getPredicate();
    }

    @Override
    public Update whereJobStatus(JobStatus jobStatus) {
        if (whereJobStatus != null) {
            throw new IllegalStateException("Job where status already set");
        }
        whereJobStatus = jobStatus;
        find.whereJobStatus(jobStatus);
        return this;
    }

    @Override
    public Update whereNextRunLeNow() {
        find.whereNextRunLeNow();
        return this;
    }

    @Override
    public Update whereTaskId(TaskId taskId) {
        find.whereTaskId(taskId);
        return this;
    }

    @Override
    public Update setNextRun(Instant nextRun) {
        setInternal(SCHEDULED_TASKS.NEXT_RUN, nextRun);
        return this;
    }

    @Override
    public Update setJobStatus(JobStatus jobStatus) {
        if (setJobStatus != null) {
            throw new IllegalStateException("Job new status already set");
        }
        setJobStatus = jobStatus;
        setFieldsFromStatus(jobStatus);
        return this;
    }

    @Override
    public Update pingJob() {
        return this.setJobStatus(JobStatus.LOCKED);
    }

    @Override
    public Update setTaskProcessingResult(TaskProcessingResult taskProcessingResult) {
        setInternal(SCHEDULED_TASKS.LAST_START_TIME, taskProcessingResult.lastStartTime());
        setStatement.set(SCHEDULED_TASKS.LAST_FINISH_TIME, taskProcessingResult.lastFinishTime());
        return this;
    }

    @Override
    public Update setTaskId(TaskId taskId) {
        setInternal(SCHEDULED_TASKS.NAME, taskId.name());
        setStatement.set(SCHEDULED_TASKS.PARAMS, taskId.param());
        return this;
    }


    Boolean getNeedReschedule() {
        return needReschedule;
    }

    @Override
    public int execute() {
        checkStatusTransition();
        return storage.executeUpdate(this);
    }

    @Override
    public Update setNeedReschedule(boolean b) {
        setInternal(SCHEDULED_TASKS.NEED_RESCHEDULE, b);
        return this;
    }

    @Override
    public Update whereNeedReschedule(boolean b) {
        find.whereNeedReschedule(b);
        return this;
    }

    UpdateBuilder.SetStatement getSetStatement() {
        return setStatement;
    }

    private void checkStatusTransition() {
        if (setJobStatus != null && whereJobStatus != null && !whereJobStatus.getAllowedTransitions().contains(setJobStatus)) {
            throw new IllegalArgumentException("Transition isn't allowed");
        }
    }

    private void setFieldsFromStatus(JobStatus status) {
        switch (status) {
            case ARCHIVED:
                setInternal(SCHEDULED_TASKS.STATUS, DELETED.getDbString());
                setStatement.setNull(SCHEDULED_TASKS.INSTANCE_ID);
                setStatement.setNull(SCHEDULED_TASKS.HEARTBEAT_TIME);
                break;

            case READY:
                setInternal(SCHEDULED_TASKS.STATUS, NEW.getDbString());
                setStatement.setNull(SCHEDULED_TASKS.INSTANCE_ID);
                setStatement.setNull(SCHEDULED_TASKS.HEARTBEAT_TIME);
                break;

            case LOCKED:
                setInternal(SCHEDULED_TASKS.STATUS, RUNNING.getDbString());
                setStatement.set(SCHEDULED_TASKS.HEARTBEAT_TIME, Instant.now());
                setStatement.set(SCHEDULED_TASKS.INSTANCE_ID, instanceId.toString());
                break;

            case STOPPED:
                setInternal(SCHEDULED_TASKS.STATUS, PAUSED.getDbString());
                setStatement.setNull(SCHEDULED_TASKS.INSTANCE_ID);
                setStatement.setNull(SCHEDULED_TASKS.HEARTBEAT_TIME);
                break;

            case EXPIRED:
                throw new IllegalArgumentException("Moving to EXPIRED state by hands is strictly prohibited by UNA");
            default:
                throw new IllegalArgumentException("Unsupported status " + status);
        }
    }

    private <T> void setInternal(Column<T> column, T value) {
        if (setStatement != null) {
            setStatement.set(column, value);
        } else {
            setStatement = set(column, value);
        }
    }

}

