package ru.yandex.direct.hourglass;

import java.time.Duration;
import java.time.temporal.TemporalUnit;

public class HourglassProperties {
    private final Duration taskFetchingInterval;
    private final Duration reschedulingTasksInterval;
    private final Duration taskHeartbeatExpiration;
    private final Duration pingInterval;
    private final Duration jobsFreeingInterval;
    private final Duration instancePingInterval;
    private final Duration scheduleUpdateInterval;
    private final Duration schedulerHeartbeatExpiration;
    private final Duration missedTaskThreshold;
    private final int threadPoolSize;
    private final int leaderVotingLowerBound;

    private HourglassProperties(Duration newTaskFetchingInterval, Duration reschedulingTasksInterval,
            Duration taskHeartbeatExpiration,
            Duration pingInterval, Duration jobsFreeingInterval, Duration instancePingInterval,
            Duration scheduleUpdateInterval, Duration schedulerHeartbeatExpiration,
            Duration missedTaskThreshold, Integer threadPoolSize, int leaderVotingLowerBound) {
        this.taskFetchingInterval = newTaskFetchingInterval;
        this.reschedulingTasksInterval = reschedulingTasksInterval;
        this.taskHeartbeatExpiration = taskHeartbeatExpiration;
        this.pingInterval = pingInterval;
        this.jobsFreeingInterval = jobsFreeingInterval;
        this.instancePingInterval = instancePingInterval;
        this.scheduleUpdateInterval = scheduleUpdateInterval;
        this.schedulerHeartbeatExpiration = schedulerHeartbeatExpiration;
        this.missedTaskThreshold = missedTaskThreshold;
        this.threadPoolSize = threadPoolSize;
        this.leaderVotingLowerBound = leaderVotingLowerBound;
    }

    public static Builder builder() {
        return new Builder();
    }

    public int getThreadPoolSize() {
        return threadPoolSize;
    }

    public Duration getTaskFetchingInterval() {
        return taskFetchingInterval;
    }

    public Duration getReschedulingTasksInterval() {
        return reschedulingTasksInterval;
    }

    public Duration getTaskHeartbeatExpiration() {
        return taskHeartbeatExpiration;
    }

    public Duration getJobsFreeingInterval() {
        return jobsFreeingInterval;
    }

    public Duration getPingInterval() {
        return pingInterval;
    }

    public Duration getInstancePingInterval() {
        return instancePingInterval;
    }

    public Duration getScheduleUpdateInterval() {
        return scheduleUpdateInterval;
    }

    public Duration getSchedulerHeartbeatExpiration() {
        return schedulerHeartbeatExpiration;
    }

    public Duration getMissedTaskThreshold() {
        return missedTaskThreshold;
    }

    public int getLeaderVotingLowerBound() {
        return leaderVotingLowerBound;
    }

    public static class Builder {

        private Duration taskFetchingInterval = Duration.ofMillis(100L);
        private Duration reschedulingTasksInterval = Duration.ofSeconds(1L);
        private Duration maxHeartbeatAge = Duration.ofSeconds(60);
        private Duration pingInterval = Duration.ofMillis(2500L);
        private Duration jobsFreeingInterval = Duration.ofSeconds(15L);
        private Duration instancePingInterval = Duration.ofMillis(7500L);
        private Duration scheduleUpdateInterval = Duration.ofMinutes(1);
        private Duration scheduleHeartbeatExpiration = Duration.ofMinutes(1);
        private Duration missedTaskThreshold = Duration.ofSeconds(90);
        private int threadPoolSize = 20;
        private int leaderVotingLowerBound = 1;

        private Builder() {
        }

        public Builder setTaskFetchingInterval(long amount, TemporalUnit temporalUnit) {
            this.taskFetchingInterval = Duration.of(amount, temporalUnit);
            return this;
        }

        public Builder setReschedulingTasksInterval(long amount, TemporalUnit temporalUnit) {
            this.reschedulingTasksInterval = Duration.of(amount, temporalUnit);
            return this;
        }

        public Builder setMaxHeartbeatAge(long amount, TemporalUnit temporalUnit) {
            this.maxHeartbeatAge = Duration.of(amount, temporalUnit);
            return this;
        }

        public Builder setPingInterval(long amount, TemporalUnit temporalUnit) {
            this.pingInterval = Duration.of(amount, temporalUnit);
            return this;
        }

        public Builder setJobsFreeingInterval(long amount, TemporalUnit temporalUnit) {
            this.jobsFreeingInterval = Duration.of(amount, temporalUnit);
            return this;
        }

        public Builder setThreadPoolSize(int threadPoolSize) {
            this.threadPoolSize = threadPoolSize;
            return this;
        }

        public Builder setInstancePingInterval(long amount, TemporalUnit temporalUnit) {
            this.instancePingInterval = Duration.of(amount, temporalUnit);
            return this;
        }

        public Builder setScheduleUpdateInterval(long amount, TemporalUnit temporalUnit) {
            this.scheduleUpdateInterval = Duration.of(amount, temporalUnit);
            return this;
        }

        public Builder setScheduleHeartbeatExpiration(long amount, TemporalUnit temporalUnit) {
            this.scheduleHeartbeatExpiration = Duration.of(amount, temporalUnit);
            return this;
        }

        public Builder setLeaderVotingLowerBound(int leaderVotingLowerBound) {
            this.leaderVotingLowerBound = leaderVotingLowerBound;
            return this;
        }

        public Builder setMissedTaskThreshold(long amount, TemporalUnit temporalUnit) {
            this.missedTaskThreshold = Duration.of(amount, temporalUnit);
            return this;
        }

        public HourglassProperties build() {
            return new HourglassProperties(taskFetchingInterval, reschedulingTasksInterval, maxHeartbeatAge,
                    pingInterval, jobsFreeingInterval, instancePingInterval, scheduleUpdateInterval,
                    scheduleHeartbeatExpiration, missedTaskThreshold, threadPoolSize, leaderVotingLowerBound);
        }
    }

}
