package ru.yandex.direct.hourglass.client;

import java.time.Instant;
import java.time.ZoneOffset;
import java.util.Collection;
import java.util.List;

import ru.yandex.direct.hourglass.storage.Job;
import ru.yandex.direct.hourglass.storage.JobStatus;
import ru.yandex.direct.hourglass.storage.Storage;
import ru.yandex.direct.hourglass.updateschedule.SchedulerInstancesRepository;

import static java.util.stream.Collectors.toList;

public class HourglassClientImpl implements HourglassClient {
    private final Storage storage;
    private final SchedulerInstancesRepository schedulerInstancesRepository;

    public HourglassClientImpl(Storage storage, SchedulerInstancesRepository schedulerInstancesRepository) {
        this.storage = storage;
        this.schedulerInstancesRepository = schedulerInstancesRepository;
    }

    @Override
    public Collection<Job> getTasks() {
        return storage.find().findJobs().stream()
                .filter(el -> el.jobStatus() != JobStatus.ARCHIVED)
                .collect(toList());
    }

    @Override
    public List<SchedulerInstance> getSchedulers() {
        return schedulerInstancesRepository.getSchedulerInstancesInfo();
    }

    @Override
    public void changeNextRunTime(Job job, Instant instant, ZoneOffset zoneOffset) {
        storage.update()
                .whereJobStatus(JobStatus.READY)
                .wherePrimaryIdIn(List.of(job.primaryId()))
                .setNextRun(instant)
                .setNeedReschedule(false)
                .execute();
    }

    @Override
    public void pauseTask(Job job) {
        storage.update()
                .wherePrimaryIdIn(List.of(job.primaryId()))
                .setJobStatus(JobStatus.STOPPED)
                .execute();
    }

    @Override
    public void continueTask(Job job) {
        storage.update()
                .wherePrimaryIdIn(List.of(job.primaryId()))
                .whereJobStatus(JobStatus.STOPPED)
                .setJobStatus(JobStatus.READY)
                .setNeedReschedule(true)
                .execute();
    }
}
