package ru.yandex.direct.hourglass.implementations;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.hourglass.HourglassProperties;
import ru.yandex.direct.hourglass.InstanceId;
import ru.yandex.direct.hourglass.MonitoringWriter;
import ru.yandex.direct.hourglass.RandomChooser;
import ru.yandex.direct.hourglass.SchedulerInstancePinger;
import ru.yandex.direct.hourglass.SchedulerService;
import ru.yandex.direct.hourglass.TaskHooks;
import ru.yandex.direct.hourglass.TaskThreadPool;
import ru.yandex.direct.hourglass.implementations.internal.ExpiredLocksRemover;
import ru.yandex.direct.hourglass.implementations.internal.LocksUpdater;
import ru.yandex.direct.hourglass.implementations.internal.NewTasksRunner;
import ru.yandex.direct.hourglass.implementations.internal.SchedulerPingerRunner;
import ru.yandex.direct.hourglass.implementations.internal.SystemThread;
import ru.yandex.direct.hourglass.implementations.internal.TasksRescheduler;
import ru.yandex.direct.hourglass.storage.Job;
import ru.yandex.direct.hourglass.storage.PrimaryId;
import ru.yandex.direct.hourglass.storage.Storage;
import ru.yandex.direct.hourglass.storage.TaskId;

public class SchedulerServiceImpl implements SchedulerService {
    private static final Logger logger = LoggerFactory.getLogger(SchedulerService.class);

    private final Map<TaskId, RunningTask> running;
    private final Storage storage;
    private final TaskThreadPool threadPool;
    private final InstanceId schedulerId;
    private final HourglassProperties hourglassProperties;
    private final AtomicBoolean started = new AtomicBoolean(false);
    private final AtomicBoolean shuttingDown = new AtomicBoolean(false);
    private final RandomChooser<PrimaryId> randomChooser;
    private final ThreadsHierarchy threadsHierarchy;
    private final SchedulerInstancePinger schedulerInstancePinger;

    private Function<Job, TaskHooks> taskAdaptor = null;
    private SystemThread newTaskFetcher;
    private SystemThread rescheduler;
    private SystemThread locksUpdater;
    private SystemThread stallLocksCollector;
    private SystemThread instancePingerThread;

    private MonitoringWriter monitoringWriter;

    public SchedulerServiceImpl(
            Storage storage,
            ThreadsHierarchy threadsHierarchy,
            TaskThreadPool taskThreadPool,
            InstanceId schedulerId,
            RandomChooser<PrimaryId> randomChooser,
            HourglassProperties hourglassProperties,
            SchedulerInstancePinger schedulerInstancePinger, MonitoringWriter monitoringWriter) {

        this.storage = storage;
        this.schedulerId = schedulerId;
        this.hourglassProperties = hourglassProperties;
        this.schedulerInstancePinger = schedulerInstancePinger;
        this.running = new ConcurrentHashMap<>();
        this.randomChooser = randomChooser;
        this.threadsHierarchy = threadsHierarchy;
        this.threadPool = taskThreadPool;
        this.monitoringWriter = monitoringWriter;
    }

    protected SystemThread scheduleSystemJob(Runnable command, ThreadFactory threadFactory, String name, long period,
                                             TimeUnit unit) {
        SystemThread systemThread = new SystemThread(command, threadFactory, name, period, unit, monitoringWriter);
        systemThread.start();
        return systemThread;
    }

    private void startNewTasksRunner(ThreadFactory systemThreadsFactory) {
        NewTasksRunner newTasksRunner =
                new NewTasksRunner(storage, threadPool, taskAdaptor, running, randomChooser, getInstanceId(),
                        monitoringWriter, hourglassProperties.getMissedTaskThreshold());

        newTaskFetcher = scheduleSystemJob(newTasksRunner, systemThreadsFactory, "get-new-task",
                hourglassProperties.getTaskFetchingInterval().toMillis(),
                TimeUnit.MILLISECONDS);
    }

    private void startRescheduler(ThreadFactory systemThreadsFactory) {
        TasksRescheduler tasksRescheduler = new TasksRescheduler(storage, schedulerId, taskAdaptor, monitoringWriter);

        rescheduler = scheduleSystemJob(tasksRescheduler, systemThreadsFactory, "reschedule-tasks",
                hourglassProperties.getReschedulingTasksInterval().toMillis(), TimeUnit.MILLISECONDS);
    }

    private void startLocksUpdater(ThreadFactory systemThreadsFactory) {
        LocksUpdater locksUpdaterRunnable = new LocksUpdater(running, storage, schedulerId, monitoringWriter);

        locksUpdater = scheduleSystemJob(locksUpdaterRunnable, systemThreadsFactory, "update-locks",
                hourglassProperties.getPingInterval().toMillis(), TimeUnit.MILLISECONDS);
    }

    private void startExpiredLocksRemover(ThreadFactory systemThreadsFactory) {
        ExpiredLocksRemover expiredLocksRemover = new ExpiredLocksRemover(storage, schedulerId, monitoringWriter);

        stallLocksCollector = scheduleSystemJob(expiredLocksRemover, systemThreadsFactory, "free-stall-locks",
                hourglassProperties.getJobsFreeingInterval().toMillis(), TimeUnit.MILLISECONDS);
    }

    private void startInstancePinger(ThreadFactory systemThreadsFactory) {
        instancePingerThread =
                scheduleSystemJob(new SchedulerPingerRunner(schedulerInstancePinger), systemThreadsFactory,
                        "instance-pinger",
                        hourglassProperties.getInstancePingInterval().toMillis(),
                        TimeUnit.MILLISECONDS);
    }

    @Override
    public void start() {
        if (taskAdaptor == null) {
            throw new IllegalStateException("TaskAdapter wasn't configured");
        }

        if (!started.compareAndSet(false, true)) {
            throw new IllegalArgumentException("Already started");
        }

        ThreadFactory systemThreadsFactory = threadsHierarchy.getSystemThreadFactory();

        startNewTasksRunner(systemThreadsFactory);
        startRescheduler(systemThreadsFactory);
        startLocksUpdater(systemThreadsFactory);
        startExpiredLocksRemover(systemThreadsFactory);
        startInstancePinger(systemThreadsFactory);

        logger.info("{} hourglass scheduler started", getInstanceId());
    }

    @Override
    public void stop() {
        if (!shuttingDown.compareAndSet(false, true)) {
            return;
        }

        logger.info("{}: stopping", getInstanceId());

        if (instancePingerThread != null) {
            instancePingerThread.stop();
        }
        if (newTaskFetcher != null) {
            newTaskFetcher.stop();
        }
        if (rescheduler != null) {
            rescheduler.stop();
        }
        if (stallLocksCollector != null) {
            stallLocksCollector.stop();
        }

        while (newTaskFetcher != null) {
            try {
                newTaskFetcher.await();
            } catch (InterruptedException e) {
                continue;
            }
            break;
        }

        threadPool.shutdown();

        for (RunningTask runningTask : getRunning().values()) {
            runningTask.getTaskHooks().onShutdown();
        }

        awaitTermination();

        if (locksUpdater != null) {
            locksUpdater.stop();
        }
    }

    private void awaitTermination() {
        while (true) {
            try {
                if (threadPool.awaitTermination(15, TimeUnit.SECONDS)) {
                    break;
                }

                for (RunningTask runningTask : getRunning().values()) {
                    logger.info("{}: shutting down and waiting for {}", getInstanceId(),
                            runningTask.getJob().taskId());
                }

            } catch (InterruptedException e) {
                logger.warn("Interrupted", e);
            }
        }

        logger.info("{}: shutting down complete", schedulerId);
    }

    @Override
    public void setJobFactory(Function<Job, TaskHooks> factory) {
        if (taskAdaptor != null) {
            throw new IllegalStateException("TaskAdapter was already configured");
        }

        taskAdaptor = factory;
    }

    @Override
    public InstanceId getInstanceId() {
        return schedulerId;
    }

    private Map<TaskId, RunningTask> getRunning() {
        return running;
    }

    Storage getStorage() {
        return storage;
    }

    public static class RunningTask {
        private final TaskHooks taskHooks;
        private final Job job;
        private final Future<Void> future;
        private volatile boolean finishing;

        public RunningTask(TaskHooks taskHooks, Job job, Future<Void> future) {
            this.taskHooks = taskHooks;
            this.job = job;
            this.future = future;
        }

        public boolean isFinishing() {
            return finishing;
        }

        public void setFinishing() {
            this.finishing = true;
        }

        public TaskHooks getTaskHooks() {
            return taskHooks;
        }

        public Job getJob() {
            return job;
        }

        public Future<Void> getFuture() {
            return future;
        }
    }

}
