package ru.yandex.direct.hourglass.implementations;

import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.BiConsumer;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/*
 * Класс занимается созданием тред-пулов. Нужен чтобы создавать их в одной иерархии потоков, с централизованнйо
 * обработкой
 * непойманных исключений.
 */

public class ThreadsHierarchy {
    private static final Logger logger = LoggerFactory.getLogger(ThreadsHierarchy.class);
    private static int id = 0;
    private final SafeThreadGroup rootThreadGroup;
    private final List<BiConsumer<Thread, Throwable>> uncaughtExceptionHandlerList;

    public ThreadsHierarchy() {
        this.uncaughtExceptionHandlerList = new CopyOnWriteArrayList<>();
        this.uncaughtExceptionHandlerList.add(((thread, throwable) ->
                logger.error("Uncaught exception in thread {} : {} ", thread, throwable)));
        this.rootThreadGroup = new SafeThreadGroup("hourglass-root-thread-group-" + (id++), getRootThreadGroup(),
                uncaughtExceptionHandlerList);

    }

    public void addUncaughtExceptionHandler(BiConsumer<Thread, Throwable> consumer) {
        uncaughtExceptionHandlerList.add(consumer);
    }

    private static ThreadGroup getRootThreadGroup() {
        ThreadGroup root = Thread.currentThread().getThreadGroup();
        ThreadGroup parent = root.getParent();

        while (parent != null) {
            root = parent;
            parent = parent.getParent();
        }

        return root;
    }

    public ThreadFactory getSystemThreadFactory() {
        return getThreadFactory("hourglass-system-threads", "hourglass-system-thread-", Thread.MAX_PRIORITY);
    }

    public ThreadFactory getWorkersThreadFactory() {
        return getThreadFactory("hourglass-workers", numberNameFactory("hourglass-worker-%d"), Thread.NORM_PRIORITY);
    }

    public ThreadFactory getThreadFactory(String groupName, String threadNamePrefix, int priority) {
        return getThreadFactory(groupName, () -> threadNamePrefix, priority);
    }


    public ThreadFactory getThreadFactory(String groupName, Supplier<String> threadNameFactory, int priority) {
        ThreadGroup newGroup = new ThreadGroup(rootThreadGroup, groupName);
        return r -> {
            var thr = new Thread(newGroup, r, threadNameFactory.get());
            thr.setPriority(priority);
            return thr;
        };
    }

    Supplier<String> numberNameFactory(String prefix) {
        AtomicLong counter = new AtomicLong(0);
        return () -> String.format(prefix, counter.incrementAndGet());
    }

    public static class SafeThreadGroup extends ThreadGroup {
        private final List<BiConsumer<Thread, Throwable>> uncaughtExceptionHandlerList;

        public SafeThreadGroup(String name, ThreadGroup parent,
                               List<BiConsumer<Thread, Throwable>> uncaughtExceptionHandlerList) {
            super(parent, name);
            this.uncaughtExceptionHandlerList = uncaughtExceptionHandlerList;
        }


        @Override
        public void uncaughtException(Thread t, Throwable e) {

            for (var handler : uncaughtExceptionHandlerList) {
                handler.accept(t, e);
            }

            super.uncaughtException(t, e);
        }
    }
}
