package ru.yandex.direct.hourglass.implementations.internal;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.hourglass.InstanceId;
import ru.yandex.direct.hourglass.MonitoringWriter;
import ru.yandex.direct.hourglass.implementations.SchedulerServiceImpl;
import ru.yandex.direct.hourglass.storage.Job;
import ru.yandex.direct.hourglass.storage.PrimaryId;
import ru.yandex.direct.hourglass.storage.Storage;
import ru.yandex.direct.hourglass.storage.TaskId;

import static ru.yandex.direct.hourglass.storage.JobStatus.LOCKED;

public class LocksUpdater implements Runnable {
    private static final Logger logger = LoggerFactory.getLogger(LocksUpdater.class);
    private final Map<TaskId, SchedulerServiceImpl.RunningTask> running;
    private final Storage storage;
    private final InstanceId schedulerId;
    private final MonitoringWriter vitalSignsListener;

    public LocksUpdater(
            Map<TaskId, SchedulerServiceImpl.RunningTask> running, Storage storage,
            InstanceId schedulerId, MonitoringWriter vitalSignsListener) {
        this.running = running;
        this.storage = storage;
        this.schedulerId = schedulerId;
        this.vitalSignsListener = vitalSignsListener;
    }

    @Override
    public void run() {
        Map<TaskId, SchedulerServiceImpl.RunningTask> runningJobsSnapshot = Map.copyOf(running);

        if (runningJobsSnapshot.isEmpty()) {
            return;
        }

        List<PrimaryId> primaryIds =
                runningJobsSnapshot.values().stream().map(t -> t.getJob().primaryId()).collect(Collectors.toList());

        if (primaryIds.isEmpty()) {
            return;
        }
        storage.update()
                .wherePrimaryIdIn(primaryIds)
                .whereJobStatus(LOCKED)
                .pingJob()
                .execute();

        Collection<Job> jobs = storage.find()
                .wherePrimaryIdIn(primaryIds)
                .whereJobStatus(LOCKED).findJobs();

        vitalSignsListener.tasksPinged(jobs.size());

        Set<TaskId> executingJobs = jobs.stream().map(Job::taskId).collect(Collectors.toSet());

        for (SchedulerServiceImpl.RunningTask runningTask : runningJobsSnapshot.values()) {
            if (!executingJobs.contains(runningTask.getJob().taskId())) {
                stopTask(runningTask);
            }
        }
    }

    void stopTask(SchedulerServiceImpl.RunningTask runningTask) {
        if (runningTask.isFinishing()) {
            return;
        }

        if (!runningTask.getTaskHooks().interrupt()) {
            runningTask.getFuture().cancel(true);
        }

        logger.warn("{}: lost task {}", schedulerId, runningTask.getJob());
        vitalSignsListener.lostTaskDetected();
    }

}
