package ru.yandex.direct.hourglass.implementations.internal;

import java.time.Duration;
import java.time.Instant;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.hourglass.InstanceId;
import ru.yandex.direct.hourglass.MonitoringWriter;
import ru.yandex.direct.hourglass.RandomChooser;
import ru.yandex.direct.hourglass.TaskHooks;
import ru.yandex.direct.hourglass.TaskProcessingResult;
import ru.yandex.direct.hourglass.TaskThreadPool;
import ru.yandex.direct.hourglass.implementations.SchedulerServiceImpl;
import ru.yandex.direct.hourglass.implementations.TaskProcessingResultImpl;
import ru.yandex.direct.hourglass.storage.Job;
import ru.yandex.direct.hourglass.storage.PrimaryId;
import ru.yandex.direct.hourglass.storage.Storage;
import ru.yandex.direct.hourglass.storage.TaskId;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.hourglass.storage.JobStatus.LOCKED;
import static ru.yandex.direct.hourglass.storage.JobStatus.READY;

public class NewTasksRunner implements Runnable {
    private static final Logger logger = LoggerFactory.getLogger(NewTasksRunner.class);
    private final Storage storage;
    private final TaskThreadPool threadPool;
    private final Function<Job, TaskHooks> taskAdaptor;
    private final Map<TaskId, SchedulerServiceImpl.RunningTask> running;
    private final RandomChooser<PrimaryId> randomChooser;
    private final InstanceId schedulerId;
    private final MonitoringWriter vitalSignsListener;
    private final Duration missedThreshold;

    private double locked = 0;
    private long attempts = 0;

    public NewTasksRunner(Storage storage, TaskThreadPool threadPool,
                          Function<Job, TaskHooks> taskAdaptor,
                          Map<TaskId, SchedulerServiceImpl.RunningTask> running,
                          RandomChooser<PrimaryId> randomChooser, InstanceId schedulerId,
                          MonitoringWriter vitalSignsListener, Duration missedThreshold) {
        this.storage = storage;
        this.threadPool = threadPool;
        this.taskAdaptor = taskAdaptor;
        this.running = running;
        this.randomChooser = randomChooser;
        this.schedulerId = schedulerId;
        this.vitalSignsListener = vitalSignsListener;
        this.missedThreshold = missedThreshold;
    }

    Map<TaskId, SchedulerServiceImpl.RunningTask> getRunning() {
        return running;
    }

    Storage getStorage() {
        return storage;
    }

    InstanceId getInstanceId() {
        return schedulerId;
    }

    @Override
    public void run() {
        int availableThreads = threadPool.availableThreadCount();

        if (availableThreads == 0) {
            return;
        }

        Collection<Job> jobs = lockTasks(availableThreads);

        vitalSignsListener.tasksLocked(jobs.size());

        for (Job job : jobs) {
            TaskHooks hooks;

            try {
                hooks = taskAdaptor.apply(job);
            } catch (Exception e) {
                logger.error("Applying task adaptor failed", e);
                continue;
            }

            if (hooks == null) {
                logger.error("Hooks for taskId {} are null", job.taskId());
                continue;
            }

            try {
                threadPool.execute(taskFuture -> {
                            var runningTask = new SchedulerServiceImpl.RunningTask(hooks, job, taskFuture);
                            getRunning().put(job.taskId(), runningTask);
                        },
                        () -> runTask(job, hooks)
                );
            } catch (Exception t) {
                logger.error("Task starting failed", t);
            }
        }

    }

    Void runTask(Job job, TaskHooks hooks) {

        try {
            Exception jobException = null;
            var startTime = Instant.now();

            logger.info("{}: Job {} with param {} started", getInstanceId(), job.taskId().name(), job.taskId().param());

            try {
                hooks.onStart();
                hooks.run();
                hooks.onFinish();
            } catch (Exception t) {
                StringBuilder sb = new StringBuilder();
                sb.append(getInstanceId());
                sb.append(": Job ");
                sb.append(job.taskId().name());
                sb.append("/");
                sb.append(job.taskId().param());
                sb.append(" failed. ");
                logger.error(sb.toString(), t);
                jobException = t;
            }

            running.get(job.taskId()).setFinishing();

            TaskProcessingResult newTaskProcessingResult = TaskProcessingResultImpl.builder()
                    .withLastFinishTime(Instant.now())
                    .withLastStartTime(startTime)
                    .withException(jobException)
                    .build();

            var nextRun = hooks.calculateNextRun(newTaskProcessingResult);

            getStorage().update()
                    .wherePrimaryIdIn(Collections.singleton(job.primaryId()))
                    .whereJobStatus(LOCKED) //We need test for it!
                    // We need test - this field will be updated
                    .setNextRun(nextRun)
                    .setTaskProcessingResult(newTaskProcessingResult)
                    .setJobStatus(READY)
                    .execute();

            logger.info("{}: Job {} with param {} next run {}", getInstanceId(), job.taskId().name(),
                    job.taskId().param(), nextRun);
        } catch (Throwable e) {
            StringBuilder sb = new StringBuilder();
            sb.append(getInstanceId());
            sb.append(": Job ");
            sb.append(job.taskId().name());
            sb.append(" with param ");
            sb.append(job.taskId().param());
            sb.append(" postprocessing failed with exception: ");
            logger.error(sb.toString(), e);
        } finally {
            running.remove(job.taskId());
        }

        return null;
    }

    boolean isJobMissed(Job job, Instant now) {
        Duration diff = Duration.between(job.nextRun(), now);

        if (diff.compareTo(missedThreshold) > 0) {
            return true;
        }

        return false;
    }

    Collection<PrimaryId> getMissedTasks(Collection<Job> readyJobs, int limit) {
        Instant now = Instant.now();
        return readyJobs.stream()
                .filter(el -> isJobMissed(el, now))
                .limit(limit)
                .map(Job::primaryId)
                .collect(toList());
    }

    Collection<PrimaryId> getReadyJobs(int amount) {
        // Когда мы читаем список готовых к запуску заданий, одно из изаданий может выполняться нами, но
        // другой инстанс посчитает его протухшим и переведет его в READY.
        // В этом случае могут быть разные гонки, задача может запуститься два раза.
        Set<PrimaryId> alreadyRun =
                getRunning().values().stream().map(el -> el.getJob().primaryId()).collect(Collectors.toSet());

        Collection<Job> readyJobs = getStorage().find()
                .whereJobStatus(READY)
                .whereNeedReschedule(false)
                .whereNextRunLeNow()
                .findJobs();

        List<PrimaryId> allIds =
                readyJobs.stream().map(Job::primaryId).filter(e -> !alreadyRun.contains(e)).collect(Collectors.toList());

        Set<PrimaryId> selectedIds = new HashSet<>(randomChooser.choose(allIds, amount));

        Collection<PrimaryId> missedTasks =
                getMissedTasks(readyJobs.stream()
                                .filter(e -> !alreadyRun.contains(e.primaryId()))
                                .collect(toList()),
                        amount - selectedIds.size());

        if (!missedTasks.isEmpty()) {
            selectedIds.addAll(missedTasks);
            logger.warn("Found missed tasks: {}", missedTasks);
            vitalSignsListener.missedTasks(missedTasks.size());
        }

        return selectedIds;
    }

    private Collection<Job> lockTasks(int amount) {

        Collection<PrimaryId> selectedIds = getReadyJobs(amount);

        if (selectedIds.isEmpty()) {
            return Collections.emptyList();
        }

        getStorage().update().wherePrimaryIdIn(selectedIds).whereJobStatus(READY).setJobStatus(LOCKED).execute();

        Collection<Job> jobs = getStorage().find()
                .wherePrimaryIdIn(selectedIds)
                .whereJobStatus(LOCKED)
                .findJobs();

        attempts++;
        locked += ((double) jobs.size() / selectedIds.size());

        logger.info(String.format("{}, Successful locks rate: %.2f chunk size: %d",
                ((double) jobs.size() / selectedIds.size()),
                selectedIds.size()) + " avg: " + locked / attempts, getInstanceId());

        logger.info("{}: took jobs: {}", getInstanceId(),
                jobs.stream().map(el -> el.primaryId().toString()).collect(
                        Collectors.joining(",")));

        return jobs;
    }
}
