package ru.yandex.direct.hourglass.implementations.internal;

import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.hourglass.MonitoringWriter;

public class SystemThread {
    private static final Logger logger = LoggerFactory.getLogger(SystemThread.class);
    private static final Long MAX_PERIOD_MILLIS = 10_000L;

    private final Thread thread;
    private final long initialPeriod;
    private final String name;
    private final MonitoringWriter vitalSignsListener;
    private final CountDownLatch stopped;
    private long currentPeriod;

    public SystemThread(Runnable runnable, ThreadFactory threadFactory, String name, long initialPeriod, TimeUnit unit,
                        MonitoringWriter vitalSignsListener) {
        this.vitalSignsListener = vitalSignsListener;
        this.initialPeriod = unit.toMillis(initialPeriod);
        this.currentPeriod = initialPeriod;
        this.stopped = new CountDownLatch(1);
        this.name = name;

        this.thread = threadFactory.newThread(() -> loop(runnable));
        this.thread.setName(thread.getName() + name);
        this.thread.setDaemon(false);
    }

    public void start() {
        thread.start();
    }

    public void stop() {
        stopped.countDown();
    }

    public void await() throws InterruptedException {
        thread.join(); // IS-NOT-COMPLETABLE-FUTURE-JOIN
    }

    private void loop(Runnable runnable) {
        while (stopped.getCount() == 1) {

            boolean failed = false;
            try {
                runnable.run();
                currentPeriod = initialPeriod;
            } catch (Throwable e) {
                failed = true;
                var logString = String.format("Thread %s failed, waiting for %dms", name, currentPeriod);
                logger.error(logString, e);

                vitalSignsListener.systemThreadFailed();

                if (!(e instanceof Exception)) {
                    throw e;
                }
            }

            try {
                stopped.await(currentPeriod, TimeUnit.MILLISECONDS);
            } catch (InterruptedException e) {
                throw new RuntimeException(e);
            }

            // Увеличиваем время ожидания в случае ошибок.
            // Если initialPeriod < MAX_PERIOD_MILLIS, то постепенно
            // увеличиваем currentPeriod в два раза пока не достигнем MAX_PERIOD_MILLIS.
            // Если же initialPeriod >= MAX_PERIOD_MILLIS, то используем initialPeriod, ничего не увеличиваем.
            if (failed) {
                currentPeriod = Math.max(initialPeriod, Math.min(currentPeriod * 2, MAX_PERIOD_MILLIS));
            }
        }
    }

}
