package ru.yandex.direct.hourglass.implementations.internal;

import java.util.Collection;
import java.util.Collections;
import java.util.function.Function;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.hourglass.InstanceId;
import ru.yandex.direct.hourglass.MonitoringWriter;
import ru.yandex.direct.hourglass.TaskHooks;
import ru.yandex.direct.hourglass.storage.Job;
import ru.yandex.direct.hourglass.storage.Storage;

import static ru.yandex.direct.hourglass.storage.JobStatus.READY;

public class TasksRescheduler implements Runnable {
    private static final Logger logger = LoggerFactory.getLogger(TasksRescheduler.class);
    private final Storage storage;
    private final InstanceId schedulerId;
    private final Function<Job, TaskHooks> taskAdaptor;
    private final MonitoringWriter vitalSignsListener;

    public TasksRescheduler(
            Storage storage,
            InstanceId schedulerId,
            Function<Job, TaskHooks> taskAdaptor, MonitoringWriter vitalSignsListener) {
        this.storage = storage;
        this.schedulerId = schedulerId;
        this.taskAdaptor = taskAdaptor;
        this.vitalSignsListener = vitalSignsListener;
    }

    Storage getStorage() {
        return storage;
    }

    InstanceId getInstanceId() {
        return schedulerId;
    }

    @Override
    public void run() {
        Collection<Job> jobs = getStorage().find()
                .whereJobStatus(READY)
                .whereNeedReschedule(true)
                .findJobs();
        vitalSignsListener.tasksRescheduled(jobs.size());
        for (Job job : jobs) {
            try {
                TaskHooks hooks = taskAdaptor.apply(job);

                if (hooks != null) {
                    getStorage().update()
                            .wherePrimaryIdIn(Collections.singleton(job.primaryId()))
                            .whereJobStatus(READY)
                            .whereNeedReschedule(true)
                            .setNextRun(hooks.calculateNextRun(job.taskProcessingResult()))
                            .setNeedReschedule(false)
                            .execute();


                } else {
                    logger.warn("TaskHooks not found for {}", job);
                }

            } catch (RuntimeException e) {
                logger.warn("Rescheduling failed", e);
            }
        }
    }
}
