package ru.yandex.direct.hourglass.implementations.randomchoosers;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import ru.yandex.direct.hourglass.RandomChooser;
import ru.yandex.direct.hourglass.implementations.MD5Hash;

/*
 *  Используем consistent hashing чтобы задания меньше пересекались между планировщиками. Нужно чтобы они более
 *  эффективно брали блокировки.
 *
 *  Хешируем с помощью md5 id  объектов, сортируем и ищем среди них позицию md5 хеша названия планировщика.
 *  Из отсортированных хешей формируем кольцо, и отбираем элементы по часовой стрелке которые идут после хэша названия
 *  планировщика.
 *
 *  Ожидаемый эффект - разные планировщики (с разными названиями) будут брать разные, непересекаиеся задания.
 *  Дополнительный возможный эффект что один и тот же планировщик может быть одно и то же задание чаще.
 *
 */

public class ConsistentRandomChooserImpl<T> implements RandomChooser<T> {
    private final long entityId;
    private final MD5Hash md5Hash;

    public ConsistentRandomChooserImpl(String entityId, MD5Hash md5Hash)
    {
        this.md5Hash = md5Hash;
        this.entityId = md5Hash.hash(entityId);
    }

    @Override
    public List<T> choose(Collection<T> elements, int size) {

        Map<Long, T> integerPrimaryIdMap =
                elements.stream().collect(Collectors.toMap(el -> md5Hash.hash(el.toString()), el -> el));

        List<Long> hashCodes = integerPrimaryIdMap.keySet().stream().sorted().collect(Collectors.toList());

        List<T> primaryIdList = new ArrayList<>();

        int p = Collections.binarySearch(hashCodes, entityId);

        // If entityId hash is present in the hashCodes list then advances p to the next position.
        // otherwise make p a positive insert position
        p = (p >= 0 ? p + 1 : -(p + 1)) % hashCodes.size();

        for (int i = p; i < hashCodes.size() && primaryIdList.size() < size; i++) {
            primaryIdList.add(integerPrimaryIdMap.get(hashCodes.get(i)));
        }

        for (int i = 0; i < p && primaryIdList.size() < size; i++) {
            primaryIdList.add(integerPrimaryIdMap.get(hashCodes.get(i)));
        }

        return primaryIdList;
    }
}
