package ru.yandex.direct.hourglass.implementations.randomchoosers;

import java.time.Clock;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;

import ru.yandex.direct.hourglass.InstanceId;
import ru.yandex.direct.hourglass.RandomChooser;
import ru.yandex.direct.hourglass.client.SchedulerInstance;
import ru.yandex.direct.hourglass.implementations.MD5Hash;
import ru.yandex.direct.hourglass.updateschedule.SchedulerInstancesRepository;

/*
 *   Для выбора задач используется алгоритм  https://en.wikipedia.org/wiki/Rendezvous_hashing
 *   По списку "живых" шедулеров выбираем наиболее подходящий для каждого задания.
 *   Раз в TIME_ROUND_BASE секунд хеширование меняется, и  задачи начинают распределяться по другому.
 *
 *   Ожидаемый эффект -
 *   Задания привязываются к конкретному шедулеру.
 *   Шедулеры не голодают
 *   Получаем гарантию того, что задачи выполняются на разных машинах
 *   Адаптируемся под изменяющееся число шедулеров.
 *
 */

public class RendezvousRandomChooser<T> implements RandomChooser<T> {

    private final static Duration TIME_ROUND_BASE = Duration.ofHours(3);

    private final SchedulerInstancesRepository schedulerInstancesRepository;
    private final String schedulerId;
    private final MD5Hash md5Hash;
    private final Clock clock;
    private final String currentVersion;

    public RendezvousRandomChooser(
            String currentVersion,
            SchedulerInstancesRepository schedulerInstancesRepository,
            InstanceId schedulerId, MD5Hash md5Hash)
    {
        this(currentVersion, schedulerInstancesRepository, schedulerId, md5Hash, Clock.systemUTC());
    }

    public RendezvousRandomChooser(
            String currentVersion,
            SchedulerInstancesRepository schedulerInstancesRepository,
            InstanceId schedulerId, MD5Hash md5Hash, Clock clock)
    {
        this.schedulerInstancesRepository = schedulerInstancesRepository;
        this.schedulerId = schedulerId.toString();
        this.md5Hash = md5Hash;
        this.clock = clock;
        this.currentVersion = currentVersion;
    }

    private long roundedTime() {
        long now = clock.millis();
        return now - (now % TIME_ROUND_BASE.toMillis());
    }

    private long hash(String truncatedTime, String element, String instance) {
        return md5Hash.hash(String.join(";", truncatedTime, element, instance));
    }

    private List<String> getActiveInstances() {
        return schedulerInstancesRepository.getSchedulerInstancesInfo().stream()
                .filter(SchedulerInstance::isActive)
                .filter(e -> e.getVersion().equals(currentVersion))
                .map(e -> e.getInstanceId().toString())
                .sorted()
                .collect(Collectors.toList());
    }

    @Override
    public List<T> choose(Collection<T> elements, int requestedSize) {
        Iterator<T> iterator = elements.iterator();

        String truncatedTime = roundedTime() + "";

        List<String> instances = getActiveInstances();

        List<T> forOurInstance = new ArrayList<>();

        while (forOurInstance.size() < requestedSize && iterator.hasNext()) {
            T id = iterator.next();
            String element = id.toString();

            long ourHash = hash(truncatedTime, element, schedulerId);

            String topInstance =
                    instances.stream()
                            .filter(instance -> hash(truncatedTime, element, instance) > ourHash)
                            .findFirst()
                            .orElse(null);

            if (topInstance == null) {
                forOurInstance.add(id);
            }
        }

        return forOurInstance;
    }
}
