package ru.yandex.direct.hourglass.implementations.updateschedule;

import java.time.Duration;
import java.util.List;
import java.util.concurrent.TimeUnit;

import ru.yandex.direct.hourglass.InstanceId;
import ru.yandex.direct.hourglass.MonitoringWriter;
import ru.yandex.direct.hourglass.implementations.ThreadsHierarchy;
import ru.yandex.direct.hourglass.implementations.internal.SystemThread;
import ru.yandex.direct.hourglass.storage.Storage;
import ru.yandex.direct.hourglass.updateschedule.MainScheduleVersionExtractor;
import ru.yandex.direct.hourglass.updateschedule.ScheduleUpdateService;
import ru.yandex.direct.hourglass.updateschedule.SchedulerInstancesRepository;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * Сервис для обновления всего расписания при смене версии приложения
 * Запускает поток, который с определенной частой пытается обновить расписание
 * 1) Получает эталонную версию расписания, если текущая версия приложения не равна эталонной - ставит флаг в бд, что
 * уведомлен, что не является эталонным инстансом
 * 2) Если претендует на обновление - берет эекслюзивную блокировку
 * 3) Узнает, является ли его версия сейчас лидирующей (большинство инстансов в бд с этой считают эту версию эталонной)
 * 3) Если версия лидирующая - обновляет раписание
 * 4) Если отличается - ничего не делает
 * 5) Отпускает блокировку
 */
public class ScheduleUpdateServiceImpl implements ScheduleUpdateService {
    private static final Duration DEFAULT_FREQUENCY = Duration.ofSeconds(5);

    private final Storage storage;
    private final String currentVersion;
    private final InstanceId instanceId;
    private final SystemThread scheduledThread;
    private final MainScheduleVersionExtractor mainScheduleVersionExtractor;
    private final SchedulerInstancesRepository schedulerInstancesRepository;
    private List<ScheduleRecord> scheduleRecords;

    private ScheduleUpdateServiceImpl(Storage storage, String currentVersion,
                                      InstanceId instanceId, ThreadsHierarchy threadsHierarchy,
                                      MonitoringWriter monitoringWriter,
                                      MainScheduleVersionExtractor mainScheduleVersionExtractor,
                                      SchedulerInstancesRepository schedulerInstancesRepository, Duration frequency) {
        this.storage = storage;
        this.currentVersion = currentVersion;
        this.instanceId = instanceId;
        var threadFactory = threadsHierarchy.getSystemThreadFactory();
        this.scheduledThread = new SystemThread(this::checkLeaderAndUpdateSchedule, threadFactory, "schedule-update",
                frequency.toMillis(),
                TimeUnit.MILLISECONDS, monitoringWriter);
        this.mainScheduleVersionExtractor = mainScheduleVersionExtractor;
        this.schedulerInstancesRepository = schedulerInstancesRepository;
    }

    public void init(List<ScheduleRecord> scheduleRecords) {
        this.scheduleRecords = scheduleRecords;
        this.scheduledThread.start();
    }

    void checkLeaderAndUpdateSchedule() {
        var mainVersion = mainScheduleVersionExtractor.getVersion();
        var isMainVersion = currentVersion.equals(mainVersion);

        if (!isMainVersion) {
            schedulerInstancesRepository.markInstanceAsNotMain(instanceId);
            return;
        } else {
            schedulerInstancesRepository.markInstanceAsMain(instanceId);
        }


        var isLeader = schedulerInstancesRepository.isLeaderVersion(currentVersion);
        if (isLeader) {
            storage.setNewSchedule(scheduleRecords);
        }
    }

    @Override
    public void close() {
        scheduledThread.stop();
    }

    public static class Builder {
        private Storage storage;
        private String currentVersion;
        private InstanceId instanceId;
        private MainScheduleVersionExtractor mainScheduleVersionExtractor;
        private SchedulerInstancesRepository schedulerInstancesRepository;
        private Duration frequency = DEFAULT_FREQUENCY;
        private ThreadsHierarchy threadsHierarchy;
        private MonitoringWriter monitoringWriter;

        public Builder withStorage(Storage storage) {
            this.storage = storage;
            return this;

        }

        public Builder withCurrentVersion(String currentVersion) {
            this.currentVersion = currentVersion;
            return this;
        }

        public Builder withInstanceId(InstanceId instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public Builder withMainVersionExtractor(MainScheduleVersionExtractor mainScheduleVersionExtractor) {
            this.mainScheduleVersionExtractor = mainScheduleVersionExtractor;
            return this;
        }

        public Builder withSchedulerInstancesRepository(SchedulerInstancesRepository schedulerInstancesRepository) {
            this.schedulerInstancesRepository = schedulerInstancesRepository;
            return this;
        }

        public Builder withFrequency(Duration frequency) {
            this.frequency = frequency;
            return this;
        }

        public Builder withThreadsHierarchy(ThreadsHierarchy threadsHierarchy) {
            this.threadsHierarchy = threadsHierarchy;
            return this;
        }

        public Builder withMonitoringWriter(MonitoringWriter monitoringWriter) {
            this.monitoringWriter = monitoringWriter;
            return this;
        }

        public ScheduleUpdateServiceImpl build() {
            checkArgument(storage != null, "forgotten storage");
            checkArgument(currentVersion != null, "forgotten currentVersion");
            checkArgument(instanceId != null, "forgotten instanceId");
            checkArgument(mainScheduleVersionExtractor != null, "forgotten mainScheduleVersionExtractor");
            checkArgument(schedulerInstancesRepository != null, "forgotten schedulerInstancesRepository");
            checkArgument(threadsHierarchy != null, "forgotten threadsHierarchy");
            checkArgument(monitoringWriter != null, "forgotten monitoringWriter");
            return new ScheduleUpdateServiceImpl(storage, currentVersion, instanceId, threadsHierarchy,
                    monitoringWriter, mainScheduleVersionExtractor, schedulerInstancesRepository, frequency);
        }
    }
}
