package ru.yandex.direct.hourglass.storage.implementations.memory;

import java.time.Instant;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import ru.yandex.direct.hourglass.TaskProcessingResult;
import ru.yandex.direct.hourglass.implementations.TaskProcessingResultImpl;
import ru.yandex.direct.hourglass.implementations.updateschedule.ScheduleRecord;
import ru.yandex.direct.hourglass.storage.Job;
import ru.yandex.direct.hourglass.storage.JobStatus;
import ru.yandex.direct.hourglass.storage.PrimaryId;
import ru.yandex.direct.hourglass.storage.Storage;
import ru.yandex.direct.hourglass.storage.TaskId;
import ru.yandex.direct.hourglass.storage.Update;
import ru.yandex.direct.hourglass.storage.implementations.Find;
import ru.yandex.direct.hourglass.storage.implementations.TaskIdImpl;

public class MemStorageImpl implements Storage {
    private volatile List<MutableJob> storage = new CopyOnWriteArrayList<>();

    public void clean() {
        storage.clear();
    }

    public void addJob(MutableJob job) {
        storage.add(job);
    }

    @Override
    public Find find() {
        return new FindImpl(this);
    }

    @Override
    public Update update() {
        return new UpdateImpl(this);
    }

    @Override
    public void setNewSchedule(Collection<ScheduleRecord> scheduleRecords) {
        AtomicInteger primaryId = new AtomicInteger();
        storage = scheduleRecords.stream().map(scheduleRecord -> scheduleRecordToJob(scheduleRecord,
                primaryId.incrementAndGet())).collect(Collectors.toList());
    }

    private List<MutableJob> getStorage() {
        return storage;
    }

    private MutableJob scheduleRecordToJob(ScheduleRecord scheduleRecord, int primaryId) {
        return new MutableJob()
                .setJobStatus(JobStatus.READY)
                .setNeedReschedule(true)
                .setTaskId(new TaskIdImpl(scheduleRecord.getName(), scheduleRecord.getParam()))
                .setPrimaryId(new IntegerPrimaryId(primaryId))
                .setScheduleHash(scheduleRecord.getScheduleHashSum())
                .setTaskProcessingResult(TaskProcessingResultImpl.builder().build())
                .setMeta(scheduleRecord.getMeta());
    }

    public static class UpdateImpl implements Update {
        private final MemStorageImpl storage;
        private Stream<MutableJob> stream;

        public UpdateImpl(MemStorageImpl storage) {
            this.storage = storage;
            this.stream = this.storage.getStorage().stream();
        }

        @Override
        public Update wherePrimaryIdIn(Collection<PrimaryId> primaryIds) {
            Set<PrimaryId> ids = new HashSet<>(primaryIds);
            stream = stream.filter(el -> ids.contains(el.primaryId()));
            return this;
        }

        @Override
        public Update whereJobStatus(JobStatus jobStatus) {
            stream = stream.filter(el -> el.jobStatus() == jobStatus);
            return this;
        }

        @Override
        public Update whereNextRunLeNow() {
            stream = stream.filter(el -> el.nextRun().compareTo(Instant.now()) <= 0);
            return this;
        }

        @Override
        public Update whereTaskId(TaskId taskId) {
            stream = stream.filter(el -> el.taskId().equals(taskId));
            return null;
        }

        @Override
        public Update setNextRun(Instant nextRun) {
            stream = stream.map(el -> el.setNextRun(nextRun));
            return this;
        }

        @Override
        public Update setJobStatus(JobStatus jobStatus) {
            stream = stream.map(el -> el.setJobStatus(jobStatus));
            return this;
        }

        @Override
        public Update pingJob() {
            return this.setJobStatus(JobStatus.LOCKED);
        }

        @Override
        public Update setTaskProcessingResult(TaskProcessingResult taskProcessingResult) {
            stream = stream.map(el -> el.setTaskProcessingResult(taskProcessingResult));
            return this;
        }

        @Override
        public Update setTaskId(TaskId taskId) {
            stream = stream.map(el -> el.setTaskId(taskId));
            return this;
        }

        @Override
        public int execute() {
            final int[] cnt = {0};
            stream.forEach(el -> cnt[0]++);
            return cnt[0];
        }

        @Override
        public Update setNeedReschedule(boolean b) {
            stream = stream.map(el -> el.setNeedReschedule(b));
            return this;
        }

        @Override
        public Update whereNeedReschedule(boolean b) {
            stream = stream.filter(t -> t.needReschedule() == b);
            return this;
        }
    }

    public static class FindImpl implements Find {

        private final MemStorageImpl storage;
        private Stream<MutableJob> stream;

        public FindImpl(MemStorageImpl storage) {
            this.storage = storage;
            this.stream = this.storage.getStorage().stream();
        }

        @Override
        public Find wherePrimaryIdIn(Collection<PrimaryId> primaryIds) {
            Set<PrimaryId> ids = new HashSet<>(primaryIds);
            stream = stream.filter(el -> ids.contains(el.primaryId()));
            return this;
        }

        @Override
        public Find whereJobStatus(JobStatus jobStatus) {
            stream = stream.filter(el -> el.jobStatus() == jobStatus);
            return this;
        }

        @Override
        public Find whereNextRunLeNow() {
            stream = stream.filter(el -> Instant.now().compareTo(el.nextRun()) >= 0);
            return this;
        }

        @Override
        public Find whereTaskId(TaskId taskId) {
            stream = stream.filter(el -> el.taskId().equals(taskId));
            return this;
        }

        @Override
        public Collection<Job> findJobs(int limit) {
            return stream.collect(Collectors.toList());
        }

        @Override
        public Collection<PrimaryId> findPrimaryIds(int limit) {
            return stream.map(Job::primaryId).limit(limit).collect(Collectors.toList());
        }

        @Override
        public Collection<PrimaryId> findPrimaryIds() {
            return stream.map(Job::primaryId).collect(Collectors.toList());
        }

        @Override
        public Collection<Job> findJobs() {
            return stream.collect(Collectors.toList());
        }

        @Override
        public Find whereNeedReschedule(boolean b) {
            stream = stream.filter(t -> t.needReschedule() == b);
            return this;
        }
    }

}
