package ru.yandex.direct.hourglass.updateschedule;

import java.util.List;

import ru.yandex.direct.hourglass.InstanceId;
import ru.yandex.direct.hourglass.InstanceMeta;
import ru.yandex.direct.hourglass.client.SchedulerInstance;

/**
 * Репозиторий, работающий с информацией о всех запущенных инстансах приложения
 * Каждый инстанс может быть основным или не основным. Основной инстанс претендует, что его версия является лидирующей.
 * Лидирующая версия - та, расписание которой будет использоваться для запуска задач
 * Инстансы с остальными версиями - не могут брать задачи из расписания не своей версии, они выполняют те задачи,
 * которые взяли до обновления расписания
 */
public interface SchedulerInstancesRepository {

    /**
     * Пометить, что версия инстанса не совпадает с эталонной версией
     *
     * @param instanceId интсанс приложения
     */
    void markInstanceAsNotMain(InstanceId instanceId);

    /**
     * Пометить, что версия инстанса совпадает с эталонной версией
     *
     * @param instanceId интсанс приложения
     */
    void markInstanceAsMain(InstanceId instanceId);

    /**
     * Проверить, является ли версия инстанса лидирующей версией
     *
     * @param currentVersion номер версии инстанса
     */
    boolean isLeaderVersion(String currentVersion);

    /**
     * Добавить новый инстанс
     *
     * @param instanceId   номер инстанса
     * @param version      номер версии, с которым работает инстанс
     * @param instanceMeta дополнительная информация об инстансе
     */
    void addInstance(InstanceId instanceId, String version, InstanceMeta instanceMeta);

    /**
     * Добавить новый инстанс
     *
     * @param instanceId номер инстанса
     * @param version    номер версии, с которым работает инстанс
     */
    default void addInstance(InstanceId instanceId, String version) {
        addInstance(instanceId, version, null);
    }

    /**
     * Удалить инстанс
     *
     * @param instanceId номер инстанса
     */
    void removeInstance(InstanceId instanceId);

    void pingInstance(InstanceId instanceId);

    List<SchedulerInstance> getSchedulerInstancesInfo();
}

