package ru.yandex.direct.i18n.tanker;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.Map;
import java.util.Set;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import org.reflections.Reflections;

import ru.yandex.direct.i18n.Language;
import ru.yandex.direct.i18n.bundle.TranslationBundle;
import ru.yandex.direct.utils.io.FileUtils;

import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toMap;

public class Downloader {
    private TankerWithBranch tanker;
    private Set<Language> languages;

    public Downloader(TankerWithBranch tanker, Set<Language> languages) {
        this.tanker = tanker;
        this.languages = languages;
    }

    public void downloadAll(
            String packageName,
            Path baseDir,
            Set<KeyTranslationStatus> allowedStatusSet,
            boolean allowIncompleteTranslations
    ) throws IOException {
        Map<String, String> keysetNames = new Reflections(packageName).getSubTypesOf(TranslationBundle.class)
                .stream()
                .map(Class::getCanonicalName)
                .collect(toMap(identity(), identity()));
        download(baseDir, allowedStatusSet, allowIncompleteTranslations, keysetNames);
    }

    public void download(
            Path baseDir,
            Set<KeyTranslationStatus> allowedStatusSet,
            boolean allowIncompleteTranslations,
            Map<String, String> keysetAndFileNames
    ) throws IOException {
        ObjectMapper mapper = new ObjectMapper();
        mapper.configure(SerializationFeature.ORDER_MAP_ENTRIES_BY_KEYS, true);

        for (Map.Entry<String, String> e : keysetAndFileNames.entrySet()) {
            String keysetName = e.getKey();
            String fileName = e.getValue();
            ProjectTranslations translations = tanker.getKeysetTranslations(
                    keysetName,
                    languages
            );

            if (!allowIncompleteTranslations) {
                translations.removeTranslations(translation -> !translation.isComplete());
            }
            translations.removeTranslations(translation -> !allowedStatusSet.contains(translation.getStatus()));

            for (Language language : languages) {
                Path dictionaryPath = baseDir.resolve(getDictionaryPath(fileName, language));
                Files.createDirectories(dictionaryPath.getParent());

                String json = mapper
                        .writerWithDefaultPrettyPrinter()
                        .writeValueAsString(translations.getSingleKeyset().makeDictionary(language));
                System.out.println(dictionaryPath);

                FileUtils.atomicWrite(json + "\n", dictionaryPath);
            }
        }
    }

    private Path getDictionaryPath(String filename, Language language) {
        String[] parts = filename.split("\\.");
        parts[parts.length - 1] += "." + language.getLangString() + ".json";
        return Paths.get(parts[0], Arrays.copyOfRange(parts, 1, parts.length));
    }
}
