package ru.yandex.direct.i18n.tanker;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonValue;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;

import ru.yandex.direct.i18n.I18NException;
import ru.yandex.direct.i18n.Language;
import ru.yandex.direct.i18n.TranslationRequest;

@JsonIgnoreProperties(ignoreUnknown = true)
public class KeyTranslations {
    private KeyTranslationsInfo info;
    private Map<Language, RawTranslation> translations;

    @JsonCreator
    KeyTranslations(
            @JsonProperty("info") KeyTranslationsInfo info,
            @JsonProperty("translations") Map<Language, RawTranslation> translations
    ) {
        if (translations.isEmpty()) {
            throw new I18NException("Translations list is empty");
        }

        // Переводы для ключа должны быть согласованы: либо все singular, либо все plural.
        if (
                !translations.values().stream().allMatch(RawTranslation::isPlural)
                        && !translations.values().stream().allMatch(RawTranslation::isSingular)
        ) {
            throw new I18NException(
                    "Inconsistent translations for key (some are singular and some are plural)."
            );
        }

        this.info = info;
        this.translations = translations;
    }

    @JsonValue
    public JsonNode toJson() {
        return new ObjectMapper().createObjectNode()
                .putPOJO("info", info)
                .putPOJO("translations", translations);
    }

    static KeyTranslations fromTranslationRequest(TranslationRequest request) {
        HashMap<Language, RawTranslation> translations = new HashMap<>();
        RawTranslation rawTranslation = RawTranslation.forDictionaryEntry(request.getEntry());

        translations.put(request.getLanguage(), rawTranslation);
        return new KeyTranslations(
                new KeyTranslationsInfo(request.getContext(), rawTranslation.isPlural(), ""),
                translations
        );
    }

    public KeyTranslationsInfo getInfo() {
        return info;
    }

    public List<Translation> getTranslations() {
        return translations.entrySet().stream()
                .map(entry -> new Translation(entry.getKey(), entry.getValue()))
                .collect(Collectors.toList());
    }

    public Optional<Translation> getTranslation(Language language) {
        RawTranslation rawTranslation = translations.get(language);
        if (rawTranslation == null) {
            return Optional.empty();
        } else {
            return Optional.of(new Translation(language, rawTranslation));
        }
    }

    public void removeTranslations(Function<Translation, Boolean> filter) {
        getTranslations().forEach(translation -> {
            if (filter.apply(translation)) {
                translations.remove(translation.getLanguage());
            }
        });
    }
}
