package ru.yandex.direct.i18n.tanker;

import java.io.IOException;
import java.util.Collections;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.JsonNode;

import ru.yandex.direct.i18n.Language;

@JsonIgnoreProperties(ignoreUnknown = true)
public class Keyset {
    private String name;
    private String project;
    private Language originalLanguage;
    private Set<Language> languages;

    private boolean jsonChecks;
    private boolean autoApproveOriginal;
    private boolean autoApprove;
    private boolean autoLocked;

    Keyset(String name, String project, Language originalLanguage, Set<Language> languages) {
        this(name, project, originalLanguage, languages, false, false, false, false);
    }

    @SuppressWarnings("checkstyle:parameternumber")
    Keyset(
            String name,
            String project,
            Language originalLanguage,
            Set<Language> languages,
            boolean jsonChecks,
            boolean autoApproveOriginal,
            boolean autoApprove,
            boolean autoLocked
    ) {
        this.name = name;
        this.project = project;
        this.originalLanguage = originalLanguage;
        this.languages = languages;
        this.jsonChecks = jsonChecks;
        this.autoApproveOriginal = autoApproveOriginal;
        this.autoApprove = autoApprove;
        this.autoLocked = autoLocked;
    }

    @JsonCreator
    public static Keyset fromJson(JsonNode tree) throws IOException {
        return KeysetJsonParser.parse(tree);
    }

    @JsonProperty("name")
    public String getName() {
        return name;
    }

    @JsonProperty("project")
    public String getProject() {
        return project;
    }

    @JsonProperty("original_language")
    public Language getOriginalLanguage() {
        return originalLanguage;
    }

    @JsonProperty("languages")
    public Set<Language> getLanguages() {
        return Collections.unmodifiableSet(languages);
    }

    @JsonProperty("json_checks")
    public boolean isJsonChecks() {
        return jsonChecks;
    }

    @JsonProperty("auto_approve_original")
    public boolean isAutoApproveOriginal() {
        return autoApproveOriginal;
    }

    @JsonProperty("auto_approve")
    public boolean isAutoApprove() {
        return autoApprove;
    }

    @JsonProperty("auto_locked")
    public boolean isAutoLocked() {
        return autoLocked;
    }

    public Keyset withName(String name) {
        this.name = name;
        return this;
    }

    @Override
    public String toString() {
        return "Keyset{" +
                "name='" + name + '\'' +
                ", project='" + project + '\'' +
                ", originalLanguage=" + originalLanguage +
                ", languages=" + languages +
                ", jsonChecks=" + jsonChecks +
                ", autoApproveOriginal=" + autoApproveOriginal +
                ", autoApprove=" + autoApprove +
                ", autoLocked=" + autoLocked +
                '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        Keyset keyset = (Keyset) o;

        if (jsonChecks != keyset.jsonChecks) {
            return false;
        }
        if (autoApproveOriginal != keyset.autoApproveOriginal) {
            return false;
        }
        if (autoApprove != keyset.autoApprove) {
            return false;
        }
        if (autoLocked != keyset.autoLocked) {
            return false;
        }
        if (!name.equals(keyset.name)) {
            return false;
        }
        if (!project.equals(keyset.project)) {
            return false;
        }
        if (originalLanguage != keyset.originalLanguage) {
            return false;
        }
        return languages.equals(keyset.languages);

    }

    @Override
    public int hashCode() {
        int result = name.hashCode();
        result = 31 * result + project.hashCode();
        result = 31 * result + originalLanguage.hashCode();
        result = 31 * result + languages.hashCode();
        result = 31 * result + (jsonChecks ? 1 : 0);
        result = 31 * result + (autoApproveOriginal ? 1 : 0);
        result = 31 * result + (autoApprove ? 1 : 0);
        result = 31 * result + (autoLocked ? 1 : 0);
        return result;
    }
}
