package ru.yandex.direct.i18n.tanker;

import java.util.Collections;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonValue;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;

import ru.yandex.direct.i18n.Language;
import ru.yandex.direct.i18n.dict.Dictionary;
import ru.yandex.direct.i18n.dict.DictionaryEntry;

@JsonIgnoreProperties(ignoreUnknown = true)
public class KeysetTranslations {
    private Map<String, KeyTranslations> keys;
    private KeysetTranslationsMeta meta;

    @JsonCreator
    KeysetTranslations(
            @JsonProperty("keys") Map<String, KeyTranslations> keys,
            @JsonProperty("meta") KeysetTranslationsMeta meta
    ) {
        this.keys = keys;
        this.meta = meta;
    }

    @JsonValue
    public JsonNode toJson() {
        return new ObjectMapper().createObjectNode()
                .putPOJO("keys", keys)
                .putPOJO("meta", meta);
    }

    public Map<String, KeyTranslations> getKeyTranslationsMap() {
        return Collections.unmodifiableMap(keys);
    }

    public KeysetTranslationsMeta getMeta() {
        return meta;
    }

    <E extends DictionaryEntry> Dictionary<E> makeDictionary(Language language) {
        if (!meta.getLanguages().contains(language)) {
            throw new IllegalArgumentException(
                    "Language '" + language.getLangString() + "' is not supported by keyset."
            );
        } else {
            Dictionary<E> dictionary = new Dictionary<>();

            for (Map.Entry<String, KeyTranslations> entry : getKeyTranslationsMap().entrySet()) {
                Optional<Translation> translation = entry.getValue().getTranslation(language);
                if (translation.isPresent()) {
                    dictionary.put(entry.getKey(), translation.get().asDictionaryEntry());
                }
            }

            return dictionary;
        }
    }

    public void removeTranslations(Function<Translation, Boolean> filter) {
        keys.forEach((keyName, translations) -> translations.removeTranslations(filter));
    }
}
