package ru.yandex.direct.i18n.tanker;

import java.util.Collections;
import java.util.Map;
import java.util.function.Function;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonValue;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;

@JsonIgnoreProperties(ignoreUnknown = true)
public class ProjectTranslations {
    private ExportInfo exportInfo;
    private Map<String, KeysetTranslations> keysets;

    @JsonCreator
    ProjectTranslations(
            @JsonProperty("export_info") ExportInfo exportInfo,
            @JsonProperty("keysets") Map<String, KeysetTranslations> keysets
    ) {
        this.exportInfo = exportInfo;
        this.keysets = keysets;

    }

    @JsonValue
    public JsonNode toJson() {
        return new ObjectMapper().createObjectNode()
                .putPOJO("export_info", exportInfo)
                .putPOJO("keysets", keysets);
    }

    public ExportInfo getExportInfo() {
        return exportInfo;
    }

    public Map<String, KeysetTranslations> getKeysets() {
        return Collections.unmodifiableMap(keysets);
    }

    public KeysetTranslations getKeyset(String keysetName) {
        return Preconditions.checkNotNull(
                keysets.get(keysetName),
                "Keyset does not exist: " + keysetName
        );
    }

    /**
     * ProjectTranslations часто используется как результат вызова Tanker.getKeysetTranslations,
     * который возвращает ProjectTranslations содержащий только один KeysetTranslations.
     * В этом случае код выглядит корявенько, потому что приходиться писать:
     *
     * <pre>tanker.getKeysetTranslations(keyset).getKeyset(keyset)</pre>
     * <p>
     * Двойное упоминание keyset сбивает с толку. В таких случаях можно использовать getSingleKeyset.
     *
     * @return Если есть ровно один keyset, возвращает его, иначе бросает исключение.
     */
    public KeysetTranslations getSingleKeyset() {
        if (keysets.size() == 1) {
            return keysets.values().iterator().next();
        } else {
            throw new IllegalStateException("Expected exactly one keyset, found " + keysets.size());
        }
    }

    public void removeTranslations(Function<Translation, Boolean> filter) {
        keysets.forEach((keysetName, translations) -> translations.removeTranslations(filter));
    }
}
