package ru.yandex.direct.i18n.tanker;

import java.text.SimpleDateFormat;
import java.util.Date;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonValue;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;

import ru.yandex.direct.i18n.I18NException;
import ru.yandex.direct.i18n.dict.DictionaryEntry;
import ru.yandex.direct.i18n.dict.PluralEntry2Form;
import ru.yandex.direct.i18n.dict.PluralEntry3Form;
import ru.yandex.direct.i18n.dict.SingularEntry;

public class RawTranslation {
    private KeyTranslationStatus status;
    private String translatorComment;
    private String author;
    private Date changeDate;
    private String form;
    private String form1;
    private String form2;
    private String form3;
    private String form4;

    @JsonCreator
    @SuppressWarnings("checkstyle:parameternumber")
    public RawTranslation(
            @JsonProperty("status") KeyTranslationStatus status,
            @JsonProperty("translator_comment") String translatorComment,
            @JsonProperty("author") String author,
            @JsonProperty("change_date") @JsonFormat(pattern = "HH:mm:ss dd.MM.yyyy") Date changeDate,
            @JsonProperty("form") String form,
            @JsonProperty("form1") String form1,
            @JsonProperty("form2") String form2,
            @JsonProperty("form3") String form3,
            @JsonProperty("form4") String form4
    ) {
        boolean emptySingular = (form == null);
        boolean emptyPlural = (form1 == null && form2 == null && form3 == null && form4 == null);
        boolean completePlural = (form1 != null && form2 != null && form3 != null && form4 != null);

        if (!emptySingular && !emptyPlural) {
            throw new I18NException("Mixed plural and singular forms");
        }
        if (emptySingular && emptyPlural) {
            throw new I18NException("Missing forms");
        }
        if (emptySingular && !completePlural) {
            throw new I18NException("Incomplete plural forms");
        }

        setStatus(status);
        this.translatorComment = translatorComment;
        this.author = author;
        this.changeDate = changeDate;
        this.form = form;
        this.form1 = form1;
        this.form2 = form2;
        this.form3 = form3;
        this.form4 = form4;
    }

    @JsonValue
    public JsonNode toJson() {
        ObjectNode tree = new ObjectMapper().createObjectNode();
        tree.putPOJO("status", status);
        if (translatorComment != null) {
            tree.put("translator_comment", translatorComment);
        }
        if (author != null) {
            tree.put("author", author);
        }
        if (changeDate != null) {
            tree.put("change_date", new SimpleDateFormat("HH:mm:ss dd.MM.yyyy").format(changeDate));
        }
        if (isSingular()) {
            tree.put("form", form);
        } else {
            tree.put("form1", form1);
            tree.put("form2", form2);
            tree.put("form3", form3);
            tree.put("form4", form4);
        }
        return tree;
    }

    public static RawTranslation forDictionaryEntry(DictionaryEntry entry) {
        if (entry instanceof SingularEntry) {
            return new RawTranslation(
                    KeyTranslationStatus.REQUIRES_TRANSLATION,
                    null,
                    null,
                    null,
                    ((SingularEntry) entry).getForm(),
                    null,
                    null,
                    null,
                    null
            );
        } else if (entry instanceof PluralEntry2Form) {
            return new RawTranslation(
                    KeyTranslationStatus.REQUIRES_TRANSLATION,
                    null,
                    null,
                    null,
                    null,
                    ((PluralEntry2Form) entry).getOne(),
                    ((PluralEntry2Form) entry).getMany(),
                    "",
                    ""
            );
        } else if (entry instanceof PluralEntry3Form) {
            return new RawTranslation(
                    KeyTranslationStatus.REQUIRES_TRANSLATION,
                    null,
                    null,
                    null,
                    null,
                    ((PluralEntry3Form) entry).getOne(),
                    ((PluralEntry3Form) entry).getSome(),
                    ((PluralEntry3Form) entry).getMany(),
                    ""
            );
        } else {
            throw new IllegalArgumentException("Don't know how to handle " + entry);
        }
    }

    public KeyTranslationStatus getStatus() {
        return status;
    }

    public String getTranslatorComment() {
        return translatorComment;
    }

    public String getAuthor() {
        return author;
    }

    public Date getChangeDate() {
        return changeDate;
    }

    public String getForm() {
        return form;
    }

    public String getForm1() {
        return form1;
    }

    public String getForm2() {
        return form2;
    }

    public String getForm3() {
        return form3;
    }

    public String getForm4() {
        return form4;
    }

    public boolean isPlural() {
        return (form == null);
    }

    public boolean isSingular() {
        return (form != null);
    }

    public void setStatus(KeyTranslationStatus status) {
        if (status == null) {
            throw new IllegalArgumentException("status can't be null");
        }
        this.status = status;
    }

    public void mergeForm(String other) {
        validateSetForm(form, other);
        if (form.isEmpty()) {
            form = other;
        }
    }

    public void mergeForm1(String other) {
        validateSetForm(form1, other);
        if (form1.isEmpty()) {
            form1 = other;
        }
    }

    public void mergeForm2(String other) {
        validateSetForm(form2, other);
        if (form2.isEmpty()) {
            form2 = other;
        }
    }

    public void mergeForm3(String other) {
        validateSetForm(form3, other);
        if (form3.isEmpty()) {
            form3 = other;
        }
    }

    public void mergeForm4(String other) {
        validateSetForm(form4, other);
        if (form4.isEmpty()) {
            form4 = other;
        }
    }

    private void validateSetForm(String before, String after) {
        if ((before == null) != (after == null)) {
            throw new I18NException("Bad form replacement: " + before + " -> " + after);
        }
    }
}
