package ru.yandex.direct.i18n.tanker;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.DefaultAsyncHttpClient;
import org.asynchttpclient.Request;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;
import org.asynchttpclient.request.body.multipart.ByteArrayPart;
import org.asynchttpclient.request.body.multipart.StringPart;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.i18n.Language;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.utils.io.FileUtils;
import ru.yandex.direct.utils.io.RuntimeIoException;

import static java.nio.charset.StandardCharsets.UTF_8;
import static ru.yandex.direct.utils.JsonUtils.fromJson;
import static ru.yandex.direct.utils.JsonUtils.getTypeFactory;

/**
 * API Танкера
 */
public class Tanker implements AutoCloseable {
    public static final String TANKER_URL = "https://tanker-api.yandex-team.ru";
    // Некоторые ручки перестали работать в старом апи, для них будем ходить в новый
    // Все перевести проблематично, так как поменялся формат запроса-ответа
    // DIRECT-145249, https://st.yandex-team.ru/TANKERMIGRATION-89#60a291758f75ec2f8c035910
    public static final String NEW_TANKER_URL = "https://tanker-beta.yandex-team.ru";
    public static final String TEST_TANKER_URL = "https://tanker-api.test.yandex-team.ru";
    private String url;
    private String projectToken;
    private String project;
    private boolean isNeedClose = false;
    private boolean debug = false;
    private final AsyncHttpClient asyncHttpClient;
    private static final int HTTP_STATUS_OK = 200;
    private static final int HTTP_NOT_FOUND = 404;
    private static final Logger logger = LoggerFactory.getLogger(Tanker.class);

    private static final JavaType WITH_KIND_AND_PROJECT_BRANCH_TYPE =
            getTypeFactory().constructParametricType(WithKindAndProject.class, Branch.class);
    private static final JavaType TANKER_RESPONSE_WITH_KIND_AND_PROJECT_BRANCH_TYPE =
            getTypeFactory().constructParametricType(TankerResponse.class, WITH_KIND_AND_PROJECT_BRANCH_TYPE);
    private static final JavaType TANKER_RESPONSE_DELETE_ITEM_TYPE =
            getTypeFactory().constructParametricType(TankerResponse.class, TankerDeletedItem.class);
    private static final JavaType WITH_KIND_KEYSET_TYPE =
            getTypeFactory().constructParametricType(WithKind.class, Keyset.class);
    private static final JavaType TANKER_RESPONSE_WITH_KIND_KEYSET_TYPE =
            getTypeFactory().constructParametricType(TankerResponse.class, WITH_KIND_KEYSET_TYPE);
    private static final JavaType TANKER_RESPONSE_KEYSET_TYPE =
            getTypeFactory().constructParametricType(TankerResponse.class, Keyset.class);
    private static final JavaType LIST_KEYSET_TYPE =
            getTypeFactory().constructParametricType(TankerList.class, Keyset.class);
    private static final JavaType TANKER_RESPONSE_LIST_KEYSET_TYPE =
            getTypeFactory().constructParametricType(TankerResponse.class, LIST_KEYSET_TYPE);

    public Tanker(String url, String projectToken, String project, boolean debug, AsyncHttpClient asyncHttpClient) {
        this.url = url;
        this.projectToken = projectToken;
        this.project = project;
        this.debug = debug;
        this.asyncHttpClient = asyncHttpClient;
    }

    public Tanker(String url, String projectToken, String project, boolean debug) {
        this(url, projectToken, project, debug, new DefaultAsyncHttpClient());
        isNeedClose = true;
    }

    public static Tanker forTokenFile(String url, Path tokenFile, String project, boolean debug) {
        String token = FileUtils.slurp(Paths.get(tokenFile.toUri())).trim();
        return new Tanker(url, token, project, debug);
    }

    public TankerWithBranch withBranch(String branch) {
        return new TankerWithBranch(this, branch);
    }

    public String getUrl() {
        return url;
    }

    public String getProject() {
        return project;
    }

    public String getProjectToken() {
        return projectToken;
    }

    public void createBranch(Branch branch) {
        String path = "/admin/project/" + urlEncode(project) + "/branch/";
        String body = JsonUtils.toJson(branch);

        Request request = new RequestBuilder("POST")
                .setUrl(url + path)
                .addHeader("Content-Type", "application/json")
                .addHeader("AUTHORIZATION", "OAuth " + projectToken)
                .setBody(body)
                .build();
        Response result = executeRequestWithLogging(request);
        handleStatus(result.getStatusCode(), result.getStatusText());
        TankerResponse<WithKindAndProject<Branch>> createdBranch =
                fromJson(result.getResponseBody(), TANKER_RESPONSE_WITH_KIND_AND_PROJECT_BRANCH_TYPE);
        /*
         *Ошибку создания бранча Танкер умеет сообщать еще одним способом, с помощью {"name": null}.
         */
        if (createdBranch.hasError() || !createdBranch.getData().getValue().getName().equals(branch.getName())) {
            throw new TankerException("Tanker returned error: " + createdBranch);
        }
    }

    public void deleteBranch(String branch) {
        String path = "/admin/project/" + urlEncode(project) + "/branch/" + branch + "/";

        Request request = new RequestBuilder("DELETE")
                .setUrl(url + path)
                .addHeader("Content-Type", "application/json")
                .addHeader("AUTHORIZATION", "OAuth " + projectToken)
                .build();

        Response result = executeRequestWithLogging(request);
        handleStatus(result.getStatusCode(), result.getStatusText());
        TankerResponse<TankerDeletedItem> createdBranch =
                fromJson(result.getResponseBody(), TANKER_RESPONSE_DELETE_ITEM_TYPE);

        if (createdBranch.hasError() || !createdBranch.getData().isSuccessful()) {
            throw new TankerException("Tanker returned error: " + createdBranch);
        }
    }

    public void createKeyset(String branch, Keyset keyset) {
        String path = "/admin/project/" + urlEncode(project) + "/keyset/";
        String body = JsonUtils.toJson(keyset);
        Request request = new RequestBuilder("POST")
                .setUrl(url + path)
                .addHeader("Content-Type", "application/json")
                .addHeader("AUTHORIZATION", "OAuth " + projectToken)
                .addQueryParam("branch", branch)
                .setBody(body)
                .build();

        Response result = executeRequestWithLogging(request);
        handleStatus(result.getStatusCode(), result.getStatusText());
        TankerResponse<WithKind<Keyset>> response =
                fromJson(result.getResponseBody(), TANKER_RESPONSE_WITH_KIND_KEYSET_TYPE);
        if (response.hasError()) {
            throw new TankerException("Tanker error: " + response.getError());
        }
    }

    public void updateKeyset(String branch, String keysetName, Keyset keyset) {
        String path = "/api/v1/project/" + urlEncode(project) + "/keyset/" + keysetName + "/";


        Meta meta = new Meta(keyset.getLanguages(), keyset.isAutoApproveOriginal(), keyset.isAutoApprove(),
                keyset.getOriginalLanguage(), keyset.isJsonChecks());

        KeySetProperties properties = new KeySetProperties(branch, meta);
        String body = JsonUtils.toJson(properties);
        Request request = new RequestBuilder("PATCH")
                // DIRECT-145249, https://st.yandex-team.ru/TANKERMIGRATION-89#60a291758f75ec2f8c035910
                // после переезда танкера обновление кейсетов сломалось по старому url
                // чтоб обновление переводов работало, пофиксила конкретно эту ручку, позже надо бы перевести все на
                // новое апи
                .setUrl(NEW_TANKER_URL + path)
                .addHeader("Content-Type", "application/json")
                .addHeader("AUTHORIZATION", "OAuth " + projectToken)
                .setBody(body)
                .build();

        Response result = executeRequestWithLogging(request);
        handleStatus(result.getStatusCode(), result.getStatusText());
        TankerResponse<Keyset> response =
                fromJson(result.getResponseBody(), TANKER_RESPONSE_KEYSET_TYPE);

        if (response.hasError()) {
            throw new TankerException(
                    "Keyset update failed: error=" + response.getError() + ", keyset=" + keyset
            );
        }
    }

    public Optional<Keyset> getKeyset(String branch, String keysetName) {
        String path = "/admin/project/" + urlEncode(project) + "/keyset/" + keysetName + "/";
        Request request = new RequestBuilder("GET")
                .setUrl(url + path)
                .addHeader("Content-Type", "application/json")
                .addHeader("AUTHORIZATION", "OAuth " + projectToken)
                .addQueryParam("branch", branch)
                .build();

        Response result = executeRequestWithLogging(request);

        if (result.getStatusCode() == HTTP_NOT_FOUND) {
            return Optional.empty();
        }
        handleStatus(result.getStatusCode(), result.getStatusText());
        TankerResponse<WithKind<Keyset>> response =
                fromJson(result.getResponseBody(), TANKER_RESPONSE_WITH_KIND_KEYSET_TYPE);

        if (response.hasError()) {
            throw new TankerException("Bad response: " + response);
        } else {
            return Optional.of(response.getData().getValue());
        }
    }

    public TankerResponse<TankerList<Keyset>> listKeysets(String branch) {
        String path = "/admin/project/" + urlEncode(project) + "/keysets/";

        Request request = new RequestBuilder("GET")
                .setUrl(url + path)
                .addHeader("Content-Type", "application/json")
                .addHeader("AUTHORIZATION", "OAuth " + projectToken)
                .addQueryParam("branch", branch)
                .build();

        Response result = executeRequestWithLogging(request);
        handleStatus(result.getStatusCode(), result.getStatusText());
        return fromJson(result.getResponseBody(), TANKER_RESPONSE_LIST_KEYSET_TYPE);
    }

    public ProjectTranslations getKeysetTranslations(String branch, String keyset, Set<Language> languages) {
        /*
         * Раньше здесь была поддержка параметра status=unapproved. Согласно документации Танкера на 2016-09-20,
         * эта опция позволяет выгружать все переводы, а не только утвержденные (по умолчанию), по факту, она
         * работает не так как описано, и, даже если этот параметр не указан, выгружаются все переводы, поэтому
         * поддержка опции удалена, а вся фильтрация должна производиться клиентом, с помощью методов класса
         * ProjectTranslations.
         *
         * Переписка на эту тему: https://ml.yandex-team.ru/thread/2370000003181881997/#message2370000003184701081
         * Тикет, в котором поменялось поведение: https://st.yandex-team.ru/TANKERDEV-1391
         */
        String path = "/keysets/tjson/";
        Request request = new RequestBuilder("GET")
                .setUrl(url + path)
                .addHeader("Content-Type", "application/json")
                .addHeader("AUTHORIZATION", "OAuth " + projectToken)
                .addQueryParam("project-id", project)
                .addQueryParam("branch-id", branch)
                .addQueryParam("keyset-id", keyset)
                .addQueryParam("flat-keyset", "yes")
                .addQueryParam("all-forms", "yes")
                .addQueryParam("language",
                        languages.stream().map(Language::getLangString).collect(Collectors.joining(","))
                )
                .build();

        Response result = executeRequestWithLogging(request);
        handleStatus(result.getStatusCode(), result.getStatusText());
        return fromJson(result.getResponseBody(), ProjectTranslations.class);
    }

    public void merge(String branch, ProjectTranslations translations, Set<Language> languages) {
        byte[] translationsAsBytes;
        try {
            translationsAsBytes = new ObjectMapper().writer().writeValueAsBytes(translations);
        } catch (JsonProcessingException exc) {
            throw new TankerException("Error while processing json in method merge", exc);
        }

        for (String keysetName : translations.getKeysets().keySet()) {
            String path = "/keysets/merge/";
            Request request = new RequestBuilder("POST")
                    .setUrl(url + path)
                    .addHeader("AUTHORIZATION", "OAuth " + projectToken)
                    .addBodyPart(
                            new StringPart("project-id", project))
                    .addBodyPart(new StringPart("branch-id", branch))
                    .addBodyPart(new StringPart("keyset-id", keysetName))
                    .addBodyPart(new StringPart("format", "tjson"))
                    .addBodyPart(new StringPart("language",
                            languages.stream().map(Language::getLangString).collect(Collectors.joining(","))))
                    .addBodyPart(new ByteArrayPart("file", translationsAsBytes, "application/json", UTF_8,
                            "data.file"))
                    .build();

            Response result = executeRequestWithLogging(request);
            handleStatus(result.getStatusCode(), result.getStatusText());
        }
    }

    private static String urlEncode(String string) {
        try {
            return URLEncoder.encode(string, "UTF-8");
        } catch (UnsupportedEncodingException exc) {
            throw new TankerException("Error while encoding url", exc);
        }
    }

    @Override
    public void close() {
        if (isNeedClose) {
            try {
                asyncHttpClient.close();
            } catch (IOException e) {
                throw new RuntimeIoException(e);
            }
        }
    }

    private Response executeRequestWithLogging(Request request) {
        try {
            if (debug) {
                logger.trace(request.toString());
            }
            Response response = asyncHttpClient.executeRequest(request).get();
            if (debug) {
                logger.trace(response.toString());
            }
            return response;
        } catch (InterruptedException | ExecutionException e) {
            throw new TankerException("Error while executing request: " + request.toString(), e);
        }
    }

    private void handleStatus(int httpStatus, String httpStatusText) {
        if (httpStatus != HTTP_STATUS_OK) {
            throw new TankerException("Bad response: " + httpStatusText);
        }
    }

}
