package ru.yandex.direct.i18n.tanker;

import ru.yandex.direct.i18n.I18NException;
import ru.yandex.direct.i18n.Language;
import ru.yandex.direct.i18n.dict.DictionaryEntry;
import ru.yandex.direct.i18n.dict.PluralEntry2Form;
import ru.yandex.direct.i18n.dict.PluralEntry3Form;
import ru.yandex.direct.i18n.dict.SingularEntry;

public class Translation {
    private Language language;
    private RawTranslation raw;

    public Translation(Language language, RawTranslation raw) {
        this.language = language;
        this.raw = raw;
    }

    public Language getLanguage() {
        return language;
    }

    public KeyTranslationStatus getStatus() {
        return raw.getStatus();
    }

    public boolean isComplete() {
        return asDictionaryEntry().isComplete();
    }

    @SuppressWarnings("unchecked")
    public <E extends DictionaryEntry> E asDictionaryEntry() {
        if (raw.isSingular()) {
            return (E) new SingularEntry(raw.getForm());

        } else if (language.getNumberOfPluralForms() == 2) {
            // Здесь в танкере может быть установленна третья форма. Использовать ее для указанного
            // языка мы не можем, но и ругаться не станем, чтобы из-за косяков в танкере не ломать
            // себе процесс.
            return (E) new PluralEntry2Form(
                    raw.getForm1(),
                    raw.getForm2().isEmpty() ? raw.getForm3() : raw.getForm2(),
                    raw.getForm4().isEmpty() ? null : raw.getForm4()
            );

        } else if (language.getNumberOfPluralForms() == 3) {
            return (E) new PluralEntry3Form(
                    raw.getForm1(),
                    raw.getForm2(),
                    raw.getForm3(),
                    raw.getForm4().isEmpty() ? null : raw.getForm4()
            );

        } else {
            throw new I18NException(
                    "Unsupported number of plural forms: " + language.getNumberOfPluralForms()
                            + ", language: " + language
            );
        }
    }

    public void setStatus(KeyTranslationStatus status) {
        raw.setStatus(status);
    }

    public void mergeDictionaryEntry(DictionaryEntry entry) {
        if (entry instanceof SingularEntry) {
            raw.mergeForm(((SingularEntry) entry).getForm());

        } else if (entry instanceof PluralEntry2Form && language.getNumberOfPluralForms() == 2) {
            PluralEntry2Form plural = (PluralEntry2Form) entry;
            raw.mergeForm1(plural.getOne());
            raw.mergeForm2(plural.getMany());
            raw.mergeForm3(plural.getMany());
            plural.getZero().ifPresent(zero -> raw.mergeForm4(zero));

        } else if (entry instanceof PluralEntry3Form && language.getNumberOfPluralForms() == 3) {
            PluralEntry3Form plural = (PluralEntry3Form) entry;
            raw.mergeForm1(plural.getOne());
            raw.mergeForm2(plural.getSome());
            raw.mergeForm3(plural.getMany());
            plural.getZero().ifPresent(zero -> raw.mergeForm4(zero));

        } else {
            throw new I18NException("Can't update translation in " + language + " using " + entry);
        }
    }
}
