package ru.yandex.direct.i18n.tanker;

import java.lang.reflect.Method;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.reflections.Reflections;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.i18n.Language;
import ru.yandex.direct.i18n.bundle.AnnotationHandler;
import ru.yandex.direct.i18n.bundle.CachedMethodInterpreter;
import ru.yandex.direct.i18n.bundle.SafeMethodInterpreter;
import ru.yandex.direct.i18n.bundle.TranslationBundle;
import ru.yandex.direct.i18n.dict.DictionaryEntry;

public class Uploader {
    private static final Logger logger = LoggerFactory.getLogger(Uploader.class);

    private TankerWithBranch tanker;
    private SafeMethodInterpreter interpreter;
    private Set<Language> languages;

    public Uploader(TankerWithBranch tanker, SafeMethodInterpreter interpreter, Set<Language> languages) {
        this.tanker = tanker;
        this.interpreter = interpreter;
        this.languages = languages;
    }

    public void uploadMergeAll(String packageName) {
        Reflections reflections = new Reflections(packageName);
        reflections.getSubTypesOf(TranslationBundle.class).forEach(this::uploadMergeBundle);
    }

    public void uploadMergeBundle(Class<? extends TranslationBundle> bundleClass) {
        Map<String, KeyTranslations> bundleTranslations = getBundleTranslations(bundleClass);
        Keyset keyset = new Keyset(
                bundleClass.getCanonicalName(),
                tanker.getProject(),
                getCommonLanguage(bundleTranslations.values()),
                languages,
                false,
                true,
                true,
                false
        );

        ensureKeysetExists(keyset);

        Map<String, KeyTranslations> tankerTranslations = tanker
                .getKeysetTranslations(keyset.getName(), languages)
                .getSingleKeyset()
                .getKeyTranslationsMap();

        /*
        Перед заливкой нужно проверить совпадение stub-переводов в коде и русских переводов
        в танкере.

        Несовпадение может означать, что заглушка поменялась, иногда такие изменения
        могут быть существенными (например, изменение порядка аргументов, было "{0} больше {1}"
        стало "{1} больше {0}"), в таких случаях нужно вручную внести изменения в танкер через
        веб-интерфейс.

        Если несовпадение вызвано несущественным изменением заглушки или перевода, тоже имеет
        смысл их согласовать, чтобы не получать ворнинги в будущем.

        Есть проблема с таким решением: пока ворнингов мало, на них будут забивать "потому что
        не критично", когда их станет много - на них перестанут обращать внимание и появляется
        риск пропустить существенное изменение.

        Есть надежная альтернатива: отслеживать изменения заглушек и переводов. Если поменялась
        только заглушка - отправляем новую версию на перевод. Если поменялся только перевод -
        выдаем ворнинг. Если поменялось и то и другое - выдаем ошибку и требуем ручного вмешательства.
         */
        tankerTranslations.forEach((key, keyTranslations) -> {
            if (bundleTranslations.containsKey(key)) {
                DictionaryEntry tankerTranslation = keyTranslations
                        .getTranslation(keyset.getOriginalLanguage()).get().asDictionaryEntry();
                DictionaryEntry stubTranslation = bundleTranslations.get(key)
                        .getTranslation(keyset.getOriginalLanguage()).get().asDictionaryEntry();

                if (!tankerTranslation.equals(stubTranslation)) {
                    logger.warn(
                            "Tanker translation differs from stub: "
                                    + "keyset: " + keyset.getName() + ", " + "key: " + key + ", "
                                    + "tanker: " + tankerTranslation + ", " + "stub: " + stubTranslation
                    );
                }
            }
        });

        /*
        Эти ключи уже в танкере, повторная отправка не нужна.

        Вообще, поскольку отправляются только новые ключи, мы могли бы использовать
        метод танкера onlyAdd вместо merge, он ровно это и делает. Предпочтение было
        отдано методу merge, потому что в таком случае есть потенциальная возможность
        автоматически переотправлять на перевод изменившиеся заглушки.
         */
        tankerTranslations.keySet().forEach(bundleTranslations::remove);

        ProjectTranslations translations = new ProjectTranslations(
                new ExportInfo(
                        new ExportInfoRequest(
                                null,
                                null,
                                tanker.getProject(),
                                tanker.getBranch(),
                                keyset.getName(),
                                CustomBoolean.YES,
                                keyset.getOriginalLanguage().getLangString()
                        ),
                        tanker.getProject(),
                        tanker.getBranch()
                ),
                Collections.singletonMap(
                        keyset.getName(),
                        new KeysetTranslations(
                                bundleTranslations,
                                new KeysetTranslationsMeta(languages)
                        )
                )
        );

        tanker.merge(translations, Collections.singleton(keyset.getOriginalLanguage()));
    }

    Language getCommonLanguage(Collection<KeyTranslations> keyTranslations) {
        Set<Language> appliedLanguages = keyTranslations.stream()
                .flatMap(key -> key.getTranslations().stream().map(Translation::getLanguage))
                .collect(Collectors.toSet());

        if (appliedLanguages.size() != 1) {
            throw new IllegalArgumentException(
                    "All keys must have translations for the same exactly one language, got: " + appliedLanguages
            );
        }

        return appliedLanguages.iterator().next();
    }

    void ensureKeysetExists(Keyset keyset) {
        Optional<Keyset> existingKeyset = tanker.getKeyset(keyset.getName());

        if (!existingKeyset.isPresent()) {
            tanker.createKeyset(keyset);
        } else {
            if (!existingKeyset.get().equals(keyset)) {
                logger.info("update keyset {}", keyset);
                tanker.updateKeyset(keyset.getName(), keyset);
            }
        }
    }

    Map<String, KeyTranslations> getBundleTranslations(Class<? extends TranslationBundle> bundleClass) {
        Map<String, KeyTranslations> bundleTranslations = new HashMap<>();

        // Проверяем, что бандл работоспособен, в частности, не содержит перегруженных методов.
        CachedMethodInterpreter.forBundle(bundleClass, interpreter);

        for (Method method : bundleClass.getDeclaredMethods()) {
            Optional<AnnotationHandler> handler = interpreter.getAnnotationHandler(method);

            if (handler.isPresent()) {
                bundleTranslations.put(
                        method.getName(),
                        KeyTranslations.fromTranslationRequest(
                                handler.get().getTranslationRequest(method)
                        )
                );
            }
        }
        return bundleTranslations;
    }
}
