package ru.yandex.direct.i18n.tanker.cli;

import java.io.IOException;
import java.nio.file.Paths;

import com.fasterxml.jackson.databind.ObjectMapper;

import ru.yandex.direct.i18n.I18NBundle;
import ru.yandex.direct.i18n.Language;
import ru.yandex.direct.i18n.tanker.Downloader;
import ru.yandex.direct.i18n.tanker.KeyTranslationStatus;
import ru.yandex.direct.i18n.tanker.ProjectTranslations;
import ru.yandex.direct.i18n.tanker.Tanker;
import ru.yandex.direct.i18n.tanker.TankerWithBranch;
import ru.yandex.direct.i18n.tanker.Uploader;
import ru.yandex.direct.i18n.tanker.migration.Migration;
import ru.yandex.direct.i18n.tanker.migration.MigrationSource;
import ru.yandex.direct.jcommander.Command;
import ru.yandex.direct.jcommander.ParserWithHelp;

import static java.util.Collections.singletonMap;

public class TankerTool {
    private CommonParams commonParams;
    private DownloadCommand download;
    private UploadCommand upload;
    private MigrateCommand migrate;
    private StatusListCommand statusList;
    private ReformatCommand reformat;

    public TankerTool(CommonParams commonParams, DownloadCommand download) {
        this.commonParams = commonParams;
        this.download = download;
        this.upload = new UploadCommand();
        this.migrate = new MigrateCommand();
        this.statusList = new StatusListCommand();
        this.reformat = new ReformatCommand();
    }

    public void run(String[] args) throws IOException {
        Command command = ParserWithHelp.parseCommand(
                TankerTool.class.getCanonicalName(),
                args,
                commonParams,
                download,
                upload,
                migrate,
                statusList,
                reformat
        );

        try (TankerWithBranch tanker = makeTanker()) {
            if (command == download) {

                if (download.singleKeyset != null && download.singleFilename != null) {
                    new Downloader(tanker, Language.getValueSet()).download(
                            Paths.get(download.targetDir),
                            download.statusSet,
                            download.incomplete,
                            singletonMap(download.singleKeyset, download.singleFilename)
                    );
                } else {
                    new Downloader(tanker, Language.getValueSet()).downloadAll(
                            commonParams.packagePrefix,
                            Paths.get(download.targetDir),
                            download.statusSet,
                            download.incomplete
                    );
                }

            } else if (command == upload) {

                new Uploader(
                        tanker,
                        I18NBundle.makeSafeMethodInterpreter(),
                        Language.getValueSet()
                ).uploadMergeAll(commonParams.packagePrefix);

            } else if (command == migrate) {

                ObjectMapper mapper = new ObjectMapper();

                MigrationSource source = new MigrationSource(
                        mapper.readerFor(ProjectTranslations.class).readValue(migrate.getTheirs().toFile())
                );
                if (source.isEmpty()) {
                    throw new IllegalArgumentException(
                            "It seems like source file '" + migrate.getTheirs() + "' "
                                    + "does not have translations for more than one language. "
                                    + "Probably, you made export from Tanker for only one language."
                    );
                }

                ProjectTranslations ourTranslations = mapper.readerFor(ProjectTranslations.class)
                        .readValue(migrate.getOurs().toFile());
                if (
                        ourTranslations.getKeysets().values().stream()
                                .flatMap(keyset -> keyset.getKeyTranslationsMap().values().stream())
                                .allMatch(key -> key.getTranslations().size() <= 1)
                ) {
                    throw new IllegalArgumentException(
                            "It seems like target file '" + migrate.getOurs() + "' "
                                    + "does not have translations for more than one language. "
                                    + "Probably, you made export from Tanker for only one language."
                    );
                }

                new Migration(source).migrate(ourTranslations, migrate.completeMigrationStatus);
                mapper.writerFor(ProjectTranslations.class).withDefaultPrettyPrinter()
                        .writeValue(System.out, ourTranslations);

            } else if (command == reformat) {

                ObjectMapper mapper = new ObjectMapper();
                mapper.writerFor(ProjectTranslations.class).withDefaultPrettyPrinter().writeValue(
                        System.out,
                        mapper.readerFor(ProjectTranslations.class).readValue(
                                reformat.getTankerExportPath().toFile()
                        )
                );

            } else if (command == statusList) {

                for (KeyTranslationStatus status : KeyTranslationStatus.values()) {
                    System.out.println(status);
                }

            } else {
                throw new IllegalArgumentException("Unsupported command: " + command);
            }
        }
    }

    private TankerWithBranch makeTanker() {
        Tanker tanker;
        if (commonParams.token != null) {
            tanker = new Tanker(commonParams.url, commonParams.token, commonParams.project, commonParams.debug);
        } else {
            tanker = Tanker.forTokenFile(commonParams.url, Paths.get(commonParams.tokenFile), commonParams.project,
                    commonParams.debug);
        }
        return new TankerWithBranch(tanker, commonParams.branch);
    }
}
