package ru.yandex.direct.i18n;

import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.function.BiFunction;
import java.util.stream.Stream;

import com.google.common.collect.ImmutableList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.i18n.bundle.TranslationKey;
import ru.yandex.direct.i18n.bundle.TranslationPluralKey;

/**
 * Класс - переводчик, делегирующий перевод цепочке переводчиков.
 * Результатом перевода является первый успешный перевод
 */
public class ChainingTranslator implements Translator {
    private static final Logger logger = LoggerFactory.getLogger(ChainingTranslator.class);
    private final Locale locale;
    private final List<Translator> translationChain;

    public ChainingTranslator(Locale locale, List<Translator> translationChain) {
        if (locale == null) {
            throw new IllegalArgumentException("locale can not be null");
        }
        this.locale = locale;
        this.translationChain = ImmutableList.copyOf(translationChain);
    }

    public ChainingTranslator(Locale locale, Translator... translationChain) {
        this(locale, Arrays.asList(translationChain));
    }

    @Override
    public String translateKey(String bundleName, String key) {
        return translate(key, (t, k) -> t.translateKey(bundleName, k));
    }

    @Override
    public String translateKey(TranslationKey key) {
        return translate(key, Translator::translateKey);
    }

    @Override
    public String translatePluralKey(TranslationPluralKey key) {
        return translate(key, Translator::translatePluralKey);
    }

    @Override
    public Locale getLocale() {
        return locale;
    }

    List<Translator> getTranslationChain() {
        return translationChain;
    }

    private <K> String translate(K key, BiFunction<Translator, K, String> translate) {
        return translationChain.stream()
                .flatMap(translator -> {
                    try {
                        return Stream.of(translate.apply(translator, key));
                    } catch (NoTranslationFoundException ex) {
                        logger.warn(
                                "failed to get translation entry '{}' by {} (OK, will try fallback translator). Error message: \"{}\"",
                                key, translator, ex.getLocalizedMessage());
                        return Stream.empty();
                    }
                })
                .findFirst()
                .orElseThrow(() -> new NoTranslationFoundException("No entry for key: " + key));
    }
}
