package ru.yandex.direct.i18n;

import java.lang.reflect.Method;
import java.util.Locale;

import com.google.common.reflect.AbstractInvocationHandler;
import com.google.common.reflect.Reflection;

import ru.yandex.direct.i18n.bundle.CachedMethodInterpreter;
import ru.yandex.direct.i18n.bundle.MessageFormatHandler;
import ru.yandex.direct.i18n.bundle.MethodInterpreter;
import ru.yandex.direct.i18n.bundle.PlainStringHandler;
import ru.yandex.direct.i18n.bundle.PluralHandler;
import ru.yandex.direct.i18n.bundle.SafeMethodInterpreter;
import ru.yandex.direct.i18n.bundle.StubTranslatorFactory;
import ru.yandex.direct.i18n.bundle.TranslationBundle;
import ru.yandex.direct.i18n.bundle.UnsafeMethodInterpreter;
import ru.yandex.direct.i18n.dict.BundleDictionaries;
import ru.yandex.direct.i18n.dict.DictionaryEntry;
import ru.yandex.direct.i18n.dict.DictionaryTranslator;

public class I18NBundle {
    public static final Locale RU = new Locale.Builder().setLanguageTag("ru").build();
    public static final Locale UA = new Locale.Builder().setLanguageTag("uk").build();
    public static final Locale EN = new Locale.Builder().setLanguageTag("en").setRegion("US").build();
    public static final Locale TR = new Locale.Builder().setLanguageTag("tr").build();

    /**
     * @param bundle Интерфейс - наследник TranslationBundle
     * @param <T>    Тип интерфейса
     * @return Объект, реализующий интернационализацию описанную в bundle.
     */
    public static <T extends TranslationBundle> T implement(Class<T> bundle) {
        return implement(bundle, CachedMethodInterpreter.forBundle(
                bundle,
                makeSafeMethodInterpreter()
        ));
    }

    public static <T extends TranslationBundle> T implement(Class<T> bundle, MethodInterpreter interpreter) {
        String description = TranslationBundle.class.getSimpleName() + " for " + bundle.getName();
        //noinspection unchecked
        return Reflection.newProxy(
                bundle,
                new AbstractInvocationHandler() {
                    @Override
                    protected Object handleInvocation(Object proxy, Method method, Object[] args) throws Throwable {
                        return interpreter.interpret(method).invoke(proxy, args);
                    }

                    @Override
                    public String toString() {
                        return description;
                    }
                }
        );
    }

    public static StubTranslatorFactory makeStubTranslatorFactory() {
        return new StubTranslatorFactory(makeSafeMethodInterpreter());
    }

    public static SafeMethodInterpreter makeSafeMethodInterpreter() {
        return new SafeMethodInterpreter()
                .withAnnotationHandler(new PlainStringHandler())
                .withAnnotationHandler(new MessageFormatHandler())
                .withAnnotationHandler(new PluralHandler());
    }

    static StubTranslatorFactory makeUnsafeStubTranslatorFactory() {
        return new StubTranslatorFactory(makeUnsafeMethodInterpreter());
    }

    static UnsafeMethodInterpreter makeUnsafeMethodInterpreter() {
        return new UnsafeMethodInterpreter()
                .withAnnotationHandler(new PlainStringHandler())
                .withAnnotationHandler(new MessageFormatHandler())
                .withAnnotationHandler(new PluralHandler());
    }

    public static <E extends DictionaryEntry> MappingTranslatorFactory makeTranslatorFactory(BundleDictionaries<E> dictionaries) {

        Translator stubTranslator = I18NBundle.makeUnsafeStubTranslatorFactory().getTranslator(RU);
        return new MappingTranslatorFactory()
                .with(
                        new ChainingTranslator(RU,
                                new DictionaryTranslator<>(dictionaries.getFor(RU)),
                                stubTranslator
                        )
                )
                .with(
                        new ChainingTranslator(UA,
                                new DictionaryTranslator<>(dictionaries.getFor(UA)),
                                new DictionaryTranslator<>(dictionaries.getFor(RU)),
                                new DictionaryTranslator<>(dictionaries.getFor(EN)),
                                stubTranslator
                        )
                )
                .with(
                        new ChainingTranslator(EN,
                                new DictionaryTranslator<>(dictionaries.getFor(EN)),
                                new DictionaryTranslator<>(dictionaries.getFor(RU)),
                                stubTranslator
                        )
                )
                .with(
                        new ChainingTranslator(TR,
                                new DictionaryTranslator<>(dictionaries.getFor(TR)),
                                new DictionaryTranslator<>(dictionaries.getFor(EN)),
                                new DictionaryTranslator<>(dictionaries.getFor(RU)),
                                stubTranslator
                        )
                );
    }
}
