package ru.yandex.direct.i18n;

import java.util.Locale;

import ru.yandex.direct.i18n.bundle.TranslationKey;
import ru.yandex.direct.i18n.bundle.TranslationPluralKey;
import ru.yandex.direct.i18n.dict.PluralEntry;
import ru.yandex.direct.i18n.dict.PluralEntry2Form;
import ru.yandex.direct.i18n.dict.PluralEntry3Form;

/**
 * Переводчик на конкретный язык. Содержит в себе соответствующую локаль.
 */
public interface Translator {
    /**
     * @param key Ключ из танкера
     * @return Строка с переводом, если перевод найден
     * @throws NoTranslationFoundException если перевод не найден
     */
    String translateKey(String bundleName, String key);

    /**
     * @param key Субъект перевода
     * @return Строка с переводом, если перевод найден
     * @throws NoTranslationFoundException если перевод не найден
     */
    String translateKey(TranslationKey key);

    /**
     * Нужно, когда перевод зависит от числительного (1 -> "яблоко", 2 -> "яблока").
     *
     * @param key Субъект перевода и числительное, определяющее конкретный вариант перевода.
     * @return Строка с переводом, если перевод найден
     * @throws NoTranslationFoundException если перевод не найден
     */
    String translatePluralKey(TranslationPluralKey key);

    /**
     * @return Используемая локаль
     */
    Locale getLocale();

    static String pluralChoice(Language language, Long numeral, PluralEntry entry) {
        if (numeral == 0 && entry.getZero().isPresent()) {
            return entry.getZero().get();
        } else {
            switch (language) {
                case EN:
                case TR:
                    PluralEntry2Form entry2 = (PluralEntry2Form) entry;
                    return pluralChoiceEn(numeral, entry2.getOne(), entry2.getMany());

                case RU:
                case UK:
                    PluralEntry3Form entry3 = (PluralEntry3Form) entry;
                    return pluralChoiceRu(numeral, entry3.getOne(), entry3.getSome(), entry3.getMany());

                default:
                    throw new RuntimeException("Not supported language: " + language);
            }
        }
    }

    static <T> T pluralChoiceEn(Long numeral, T one, T many) {
        return numeral == 1 ? one : many;
    }

    /**
     * assert pluralChoice(1 ,'баннер' ,'баннера' ,'баннеров').equals('баннер');
     * assert pluralChoice(2 ,'баннер' ,'баннера' ,'баннеров').equals('баннера');
     * assert pluralChoice(5 ,'баннер' ,'баннера' ,'баннеров').equals('баннеров');
     */
    static <T> T pluralChoiceRu(Long numeral, T one, T some, T many) {
        Long lastDigit = Math.abs(numeral) % 10;
        Long prevLastDigit = (Math.abs(numeral) % 100) / 10;

        if (lastDigit > 1 && lastDigit < 5 && prevLastDigit != 1) {
            return some;
        } else if (prevLastDigit == 1 || lastDigit > 4 || lastDigit == 0) {
            return many;
        } else {
            return one;
        }
    }
}
