package ru.yandex.direct.i18n.bundle;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;

import ru.yandex.direct.i18n.TranslationRequest;

/**
 * Обработчик аннотации методов в TranslationBundle.
 * Используются в составе MethodInterpreter.
 */
public interface AnnotationHandler {
    /**
     * Для обработки аннотированных методов в TranslationBundle нужно
     * уметь понимать, какой AnnotationHandler обрабатывает соответствующую
     * аннотацию. Сопоставить обработчик и  аннтацию можно несколькими способами:
     * <p>
     * 1. описать обработчики и аннотации независимо друг от друга,
     * и в SafeMethodInterpreter передавать пары аннотация+обработчик.
     * Недостаток - есть возможность указать несовместимую пару,
     * и такую ошибку нельзя в общем случае отследить и предотвратить,
     * бросив исключение. Отказать.
     * <p>
     * 2. Указать в аннотации класс обработчика. Клевый способ,
     * потому что есть возможность указать один обработчик для разных
     * аннотаций, и при этом, естественным образом нельзя (по ошибке)
     * указать разные обработчики для одной аннотации.
     * Недостаток - можно указать только класс обработчика, но не конкретный
     * инстанс, т.е. нельзя использовать параметризованные обработчики.
     * Кажется, на нас этот недостаток прямо сейчас вообще никак не влияет,
     * но общность ограничивает, поэтому - нет.
     * <p>
     * 3. Указать в обработчике какую аннотацию он обрабатывает.
     * Общность не нарушена, но при этом можно по ошибке передать в
     * SafeMethodInterpreter несколько обработчиков, обслуживающих одну и ту же
     * аннотацию, а это вызовет ошибку во время исполнения. Отстой.
     * Тем не менее, это самый общий случай, поэтому выбор сделан в его пользу.
     *
     * @return Класс аннотации, который обрабатвыается данным AnnotationHandler.
     */
    Class<? extends Annotation> getAnnotationType();

    MethodInterpretation interpret(Method method);

    /**
     * То же, что и interpret, только без дополнительных проверок. Объясню, что значит
     * дополнительные проверки на примере:
     * <p>
     * PluralHandler требует, чтобы ровно один параметр метода был аннотирован с помощью @Plural.
     * Может случится так, что программист забыл указать @Plural, или указал, но у параметра с
     * неподходящим (не целочисленным) типом, в таком случае у PluralHandler нет шансов успешно
     * интерпретировать метод, поэтому эта проверка не является дополнительной, она происходит
     * естественным образом при поиске аннотированного параметра.
     * <p>
     * Есть другая ошибка - разработчик случайно аннотировал как @Plural несколько параметров.
     * В этом случае мы можем найти первый и использовать его, а остальные игнорировать.
     * Это не очень хорошо, клиенту может улететь неверный перевод, но это лучше чем упасть
     * с 500-ой ошибкой. Вот как раз проверка того, что @Plural указан ровно один раз и является
     * дополнительной.
     */
    MethodInterpretation unsafeInterpret(Method method);

    /**
     * Возвращает данные, необходимые для перевода в Танкере.
     *
     * @param method
     * @return
     */
    TranslationRequest getTranslationRequest(Method method);
}
