package ru.yandex.direct.i18n.bundle;

import java.lang.reflect.Method;
import java.util.HashSet;
import java.util.Set;

import com.google.common.collect.ImmutableMap;

import ru.yandex.direct.i18n.I18NException;

public class CachedMethodInterpreter implements MethodInterpreter {
    private final ImmutableMap<Method, MethodInterpretation> interpretations;

    private CachedMethodInterpreter(ImmutableMap<Method, MethodInterpretation> interpretations) {
        this.interpretations = interpretations;
    }

    public static <T extends TranslationBundle> CachedMethodInterpreter forBundle(
            Class<T> bundle, MethodInterpreter interpreter) {
        ImmutableMap.Builder<Method, MethodInterpretation> interpretations = ImmutableMap.builder();

        /*
        Для танкера ключем является имя метода, поэтому во избежание двусмысленностей
        запрещаем перегруженные методы.
         */
        Set<String> methodNames = new HashSet<>();

        for (Method method : bundle.getDeclaredMethods()) {
            if (methodNames.contains(method.getName())) {
                throw new I18NException(
                        "Overloaded methods are not allowed: "
                                + bundle + ": " + method.getName()
                );
            }
            /*
            heretic@
            Метод $jacocoInit навешивается в runtime java агентом JaCoCo( /devtools/jacoco-agent ),
            который используется для формирования отчетов по coverage. Он не должен тестироваться.
             */
            if (method.getName().endsWith("$jacocoInit")) {
                continue;
            }
            methodNames.add(method.getName());
            interpretations.put(method, interpreter.interpret(method));
        }
        return new CachedMethodInterpreter(interpretations.build());
    }

    @Override
    public MethodInterpretation interpret(Method method) {
        MethodInterpretation interpretation = interpretations.get(method);
        if (interpretation == null) {
            throw new RuntimeException("Not found interpretation for method: " + method);
        }
        return interpretation;
    }
}
