package ru.yandex.direct.i18n.bundle;

import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.MethodType;
import java.lang.invoke.WrongMethodTypeException;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import ru.yandex.direct.i18n.I18NException;
import ru.yandex.direct.i18n.Translatable;

/**
 * Интерпретация для default-методов в TranslationBundle.
 * Фактически, интерпретация заключается всего лишь в вызове самого метода,
 * а многабукав ниже - это всего лишь технические детали.
 */
public class DefaultMethodInterpretation implements MethodInterpretation {
    private MethodHandle methodHandle;

    public DefaultMethodInterpretation(MethodHandle methodHandle) {
        this.methodHandle = methodHandle;
    }

    /**
     * Возвращает интерпретацию для default-метода интерфейса.
     */
    public static DefaultMethodInterpretation forMethod(Method method) {
        final Class<?> declaringClass = method.getDeclaringClass();
        try {
            Constructor<MethodHandles.Lookup>
                    constructor = MethodHandles.Lookup.class.getDeclaredConstructor(Class.class, int.class);
            constructor.setAccessible(true);
            // -1 == TRUSTED, не проверять права на метод
            // по-хорошему стоит попробовать jOOR или подобную библиотеку
            MethodHandles.Lookup lookup = constructor.newInstance(declaringClass, -1);
            MethodHandle methodHandle = lookup.findSpecial(
                    method.getDeclaringClass(),
                    method.getName(),
                    MethodType.methodType(method.getReturnType(), method.getParameterTypes()),
                    method.getDeclaringClass()
            );
            return new DefaultMethodInterpretation(methodHandle);
        } catch (
                NoSuchMethodException
                        | IllegalAccessException
                        | InvocationTargetException
                        | InstantiationException
                        | IllegalStateException
                        exc
        ) {
            // не I18NException
            throw new IllegalStateException("Internal error, unexpected exception", exc);
        }
    }

    @Override
    public Translatable invoke(Object proxy, Object[] args) {
        /*
        В общем случае, methodHandle.invokeWithArguments() может быть вызван на каком угодно
        методе. Какой угодно метод, в общем случае, может бросить какой угодно Throwable.
        Поскольку на этапе компиляции не известно какой метод будет использоваться,
        methodHandle.invokeWithArguments() вынужден объявить, что он может бросить какой
        угодно Throwable.

        Это, в свою очередь, означает, что любой метод использующий интернационализацию,
        будет вынужден задекларировать то же самое. И тут получается, что checked-исключения
        приносят не пользу, а вред, куча методов с объявленным throws Throwable - это
        информационный шум.

        Стараемся выкрутиться, оборачивая все checked-исключения в chained RuntimeError.
         */
        try {
            Object translatable = methodHandle.bindTo(proxy).invokeWithArguments(args);
            try {
                return (Translatable) translatable;
            } catch (ClassCastException exc) {
                throw new I18NException("Method must return Translatable: " + methodHandle, exc);
            }
        } catch (IllegalArgumentException | ClassCastException | WrongMethodTypeException exc) {
            /*
            Эти исключения могут быть брошены методами bindTo и invokeWithArguments.

            С ними есть небольшая упячка, мы не можем отличить исключение брошенное методом
            invokeWithArguments от того же исключения брошенного целевым методом из methodHandle.

            Это умышленно не I18NException, потому что такое исключение означает внутреннюю
            ошибку библиотеки, а не ошибку пользователя i18n. А вообще, такое никогда не должно
            случаться.
             */
            throw new RuntimeException("Internal error", exc);
        } catch (Error | RuntimeException exc) {
            /*
            Unchecked-исключения пробрасываем как есть, чтобы упростить чтение стек-трейсов.

            Кроме этого, важно, чтобы наружу свободно пролетел обернутый chained
            ClassCastException, который может случится при касте Object в Translatable.
             */
            throw exc;
        } catch (Throwable exc) {
            /*
            Checked-исключения оборачиваем в RuntimeException.

            Это исключение, произошедшее в процессе работы целевого default-метода,
            это не ошибка связанная с неверным использованием библиотеки i18n, поэтому
            RuntimeException, а не I18NException.
             */
            throw new RuntimeException("Exception in default method: " + methodHandle, exc);
        }
    }
}
