package ru.yandex.direct.i18n.bundle;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;

import ru.yandex.direct.i18n.I18NException;
import ru.yandex.direct.i18n.TranslationRequest;
import ru.yandex.direct.i18n.dict.PluralEntry3Form;

/**
 * Обработчик для @PluralStub
 */
public class PluralHandler implements AnnotationHandler {
    @Override
    public Class<? extends Annotation> getAnnotationType() {
        return PluralStub.class;
    }

    @Override
    public MethodInterpretation interpret(Method method) {
        return new MethodInterpretationWithPluralStub(
                TranslationKey.forMethod(method),
                getPluralParameterIndex(method, false),
                method.getAnnotation(PluralStub.class)
        );
    }

    @Override
    public MethodInterpretation unsafeInterpret(Method method) {
        return new MethodInterpretationWithPluralStub(
                TranslationKey.forMethod(method),
                getPluralParameterIndex(method, true),
                method.getAnnotation(PluralStub.class)
        );
    }

    /**
     * @param ignoreAmbiguity остановить поиск на первом найденном @Plural.
     *                        Нужно выставлять в true, если не хочется падать
     *                        в продакшене на кривых методах в TranslationBundle.
     * @return Номер параметра, который нужно использовать в качестве числительного
     * для множественного перевода.
     */
    private static Integer getPluralParameterIndex(Method method, boolean ignoreAmbiguity) {
        Integer pluralParameterIndex = null;
        Parameter[] parameters = method.getParameters();
        for (int i = 0; i < parameters.length; i++) {
            if (parameters[i].getAnnotation(Plural.class) != null) {
                if (pluralParameterIndex != null) {
                    throw new I18NException("Conflicting @Plural handlers in: " + method);
                } else {
                    if (
                            long.class.isAssignableFrom(parameters[i].getType())
                                    || Long.class.isAssignableFrom(parameters[i].getType())
                    ) {
                        pluralParameterIndex = i;
                        if (ignoreAmbiguity) {
                            break;
                        }
                    } else {
                        throw new I18NException("@Plural parameter must be of integral type: " + method);
                    }
                }
            }
        }
        if (pluralParameterIndex == null) {
            throw new I18NException(
                    "One of integral parameters should be annotated with @Plural: " + method
            );
        }
        return pluralParameterIndex;
    }

    @Override
    public TranslationRequest getTranslationRequest(Method method) {
        PluralStub stub = method.getDeclaredAnnotation(PluralStub.class);
        return new TranslationRequest(
                new PluralEntry3Form(stub.one(), stub.some(), stub.many()),
                stub.language(),
                stub.comment()
        );
    }
}
