package ru.yandex.direct.i18n.bundle;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.Optional;

import ru.yandex.direct.i18n.I18NException;

public class SafeMethodInterpreter extends AbstractMethodInterpreter {
    public Optional<AnnotationHandler> getAnnotationHandler(Method method) {
        if (method.isDefault()) {
            return Optional.empty();
        } else {
            Annotation[] annotations = method.getDeclaredAnnotations();

            if (annotations.length == 0) {
                throw new I18NException("Translation method without handled annotations: " + method);

            } else if (annotations.length != 1) {
                // Взаимодействие нескольких аннотаций на одном методе не определено.
                throw new I18NException(
                        "Don't know how to handle multiple handlers for method "
                                + method + ": " + Arrays.toString(annotations)
                );

            } else {
                if (!containsAnnotation(annotations[0].annotationType())) {
                    throw new I18NException("No handler for annotation: " + annotations[0]);
                } else {
                    return Optional.of(handlerFor(annotations[0].annotationType()));
                }
            }
        }
    }

    /**
     * @param method Метод, для которого требуется вернуть интерпретацию.
     * @return MethodInterpretation, ассоциированный с методом method.
     */
    public MethodInterpretation interpret(Method method) {
        if (method.isDefault()) {
            throw new I18NException("Default methods are not supported, see DIRECT-172039");
        } else {
            //noinspection OptionalGetWithoutIsPresent
            return getAnnotationHandler(method).get().interpret(method);
        }
    }

    /**
     * @param annotationHandler Добавляемый обработчик аннотации
     * @return this с добавленным обработчиком
     */
    public SafeMethodInterpreter withAnnotationHandler(AnnotationHandler annotationHandler) {
        addAnnotationHandler(annotationHandler);
        return this;
    }
}
