package ru.yandex.direct.i18n.dict;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonValue;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Lists;

import ru.yandex.direct.i18n.I18NException;
import ru.yandex.direct.i18n.NoTranslationFoundException;

public class Dictionary<E extends DictionaryEntry> {
    static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();

    private Map<String, E> dictionary;

    public Dictionary() {
        dictionary = new HashMap<>();
    }

    @JsonCreator
    public static <E extends DictionaryEntry> Dictionary<E> fromJson(JsonNode tree) throws IOException {
        Dictionary<E> dictionary = new Dictionary<>();

        if (!Lists.newArrayList(tree.fieldNames()).equals(Collections.singletonList("dictionary"))) {
            throw new IllegalArgumentException(
                    "Expected exactly one key: \"dictionary\", got: " + tree
            );
        }

        for (
                Iterator<Map.Entry<String, JsonNode>> fields = tree.get("dictionary").fields();
                fields.hasNext();
        ) {
            Map.Entry<String, JsonNode> field = fields.next();
            E entry;

            if (field.getValue().get("is_plural").asBoolean()) {
                if (field.getValue().has("some")) {
                    entry = Dictionary.OBJECT_MAPPER.readerFor(PluralEntry3Form.class).readValue(field.getValue());
                } else {
                    entry = Dictionary.OBJECT_MAPPER.readerFor(PluralEntry2Form.class).readValue(field.getValue());
                }
            } else {
                entry = Dictionary.OBJECT_MAPPER.readerFor(SingularEntry.class).readValue(field.getValue());
            }

            dictionary.put(field.getKey(), entry);
        }

        return dictionary;
    }

    @JsonValue
    public JsonNode toJson() {
        return Dictionary.OBJECT_MAPPER.createObjectNode().putPOJO("dictionary", dictionary);
    }

    public static <E extends DictionaryEntry> Dictionary<E> fromInputStream(InputStream stream) throws IOException {
        return Dictionary.OBJECT_MAPPER.readValue(stream, new TypeReference<Dictionary<E>>() {
        });
    }

    public void put(String key, E entry) {
        if (key == null) {
            throw new I18NException("Null key");
        }
        if (entry == null) {
            throw new I18NException("Null entry for key: " + key);
        }
        E previous = dictionary.putIfAbsent(key, entry);
        if (previous != null) {
            throw new I18NException(
                    "Duplicate key " + key + ": "
                            + "previous: " + previous + ", "
                            + "new: " + entry
            );
        }
    }

    public E get(String key) {
        E entry = dictionary.get(key);
        if (entry == null) {
            throw new NoTranslationFoundException("No entry for key: " + key);
        } else {
            return entry;
        }
    }
}
