package ru.yandex.direct.i18n.dict;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.i18n.Language;
import ru.yandex.direct.i18n.StrUtil;

/**
 * Загрузчик BundleDictionary'ов. Для корректного распознавания путь к bundl'у должен быт в формате:
 * [localeDirPrefix][bundleName].[languageTag].json
 * При этом localeDirPrefix нужен для корректного распознавания bundleName; в итоговый BundleDictionary
 * он(localeDirPrefix) не попадет.
 */
public class DictionaryLoader {
    public static final String PROTOCOL_JAR = "jar";
    public static final String PROTOCOL_FILE = "file";
    public static final String JAR_CONTAINER_TO_FILE_SEPARATOR = "jar!";
    public static final char BUNDLE_TOKENS_SEPARATOR = '.';
    public static final char URL_PATH_SEPARATOR = '/';

    private static final Logger logger = LoggerFactory.getLogger(DictionaryLoader.class);

    private final String localeDirPrefix;

    public DictionaryLoader(String localeDirPrefix) {
        this.localeDirPrefix = localeDirPrefix;
    }

    public BundleDictionaries loadFromURLs(Collection<URL> urls) throws IOException {
        return loadFromSources(collectBundleDictionarySources(urls));
    }

    public static <E extends DictionaryEntry> BundleDictionaries<E> loadFromSources(Collection<BundleDictionarySource> sources) throws IOException {
        Collection<BundleDictionary<E>> result = new ArrayList<>(sources.size());
        for (BundleDictionarySource source : sources) {
            try (InputStream is = source.getFileUrl().openStream()) {
                result.add(new BundleDictionary<>(
                        source.getBundleName(),
                        source.getLanguage(),
                        Dictionary.fromInputStream(is)
                ));
            }
        }
        return new BundleDictionaries<>(result);
    }

    Collection<BundleDictionarySource> collectBundleDictionarySources(Collection<URL> urls) {
        return urls.stream()
                .map(this::makeBundleDictionarySource)
                .collect(Collectors.toList());
    }

    public BundleDictionarySource makeBundleDictionarySource(URL url) {
        String fileName = extractDictionaryPath(url);
        String[] tokens = StrUtil.rsplit(fileName, BUNDLE_TOKENS_SEPARATOR, 3);
        Language language = Language.fromLangString(tokens[1]);
        String bundleName = tokens[0].replace(URL_PATH_SEPARATOR, '.');
        return new BundleDictionarySource(bundleName, language, url);
    }

    private String extractDictionaryPath(URL url) {
        String entryName = url.getFile();
        String urlProto = url.getProtocol();
        if (PROTOCOL_JAR.equals(urlProto)) {
            //аналог ((JarURLConnection)(url.openConnection())).getEntryName(), но openConnection может генерить IOException
            //При этом Javadoc утверждает, что реальный IO будет только при URLConnection.connect()
            String[] fileTokens = entryName.split(JAR_CONTAINER_TO_FILE_SEPARATOR, 2);
            entryName = fileTokens.length == 2 ? fileTokens[1] : fileTokens[0];
        } else {
            if (!PROTOCOL_FILE.equals(urlProto)) {
                throw new IllegalArgumentException("Unsupported protocol: " + urlProto);
            }
        }
        int localeDirFirst = localeDirPrefix.isEmpty() ? 0 : entryName.indexOf(localeDirPrefix);
        int localeDirLast = localeDirPrefix.isEmpty() ? 0 : entryName.lastIndexOf(localeDirPrefix);
        if (localeDirFirst >= 0) {
            if (localeDirFirst == localeDirLast) {
                return entryName.substring(localeDirFirst + localeDirPrefix.length());
            } else {
                throw new IllegalArgumentException(url + " is not recognized as locale bundle: multiple prefixes found");
            }
        } else {
            throw new IllegalArgumentException(url + " is not recognized as locale bundle: no prefix found");
        }
    }
}

