package ru.yandex.direct.i18n.dict;

import java.util.Locale;
import java.util.Map;

import com.google.common.collect.ImmutableMap;

import ru.yandex.direct.i18n.Language;
import ru.yandex.direct.i18n.NoTranslationFoundException;
import ru.yandex.direct.i18n.Translator;
import ru.yandex.direct.i18n.bundle.TranslationKey;
import ru.yandex.direct.i18n.bundle.TranslationPluralKey;

public class DictionaryTranslator<E extends DictionaryEntry> implements Translator {
    private final Locale locale;
    /**
     * bundleName -> Dictionary; language is fixed by locale
     */
    private final Map<String, Dictionary<E>> bundleDictionaryMap;

    public DictionaryTranslator(Locale locale, Map<String, Dictionary<E>> bundleDictionaryMap) {
        this.locale = locale;
        this.bundleDictionaryMap = ImmutableMap.copyOf(bundleDictionaryMap);
    }

    public DictionaryTranslator(LocaleDictionaries<E> dictionary) {
        this(dictionary.getLocale(), dictionary.getDictionaries());
    }

    @Override
    public String translateKey(TranslationKey key) {
        Dictionary<E> dictionary = bundleDictionaryMap.get(key.bundleName());
        if (dictionary != null) {
            E entry = dictionary.get(key.toMethod().getName());
            if (entry instanceof SingularEntry) {
                return ((SingularEntry) entry).getForm();
            } else {
                throw new RuntimeException(
                        "Something is very wrong with the dictionary: " + dictionary + ". "
                                + "Expected to have singular entry for " + key + ", got " + entry + "."
                );
            }
        } else {
            throw new NoTranslationFoundException("No entry for key: " + key);
        }
    }

    @Override
    public String translateKey(String bundleName, String key) {
        Dictionary<E> dictionary = bundleDictionaryMap.get(bundleName);
        if (dictionary != null) {
            E entry = dictionary.get(key);
            if (entry instanceof SingularEntry) {
                return ((SingularEntry) entry).getForm();
            } else {
                throw new RuntimeException(
                        "Something is very wrong with the dictionary: " + dictionary + ". "
                                + "Expected to have singular entry for " + key + ", got " + entry + "."
                );
            }
        } else {
            throw new NoTranslationFoundException("No entry for key: " + key);
        }
    }

    @Override
    public String translatePluralKey(TranslationPluralKey key) {
        Dictionary<E> dictionary = bundleDictionaryMap.get(key.bundleName());
        if (dictionary != null) {
            E entry = dictionary.get(key.keyName());

            if (entry instanceof PluralEntry) {
                return Translator.pluralChoice(
                        Language.fromLocale(locale),
                        key.getNumeral(),
                        (PluralEntry) entry
                );
            } else {
                throw new RuntimeException(
                        "Something is very wrong with the dictionary: " + dictionary + ". "
                                + "Expected to have plural entry for " + key + ", got " + entry + "."
                );
            }
        } else {
            throw new NoTranslationFoundException("No entry for key: " + key);
        }
    }

    @Override
    public Locale getLocale() {
        return locale;
    }

    @Override
    public String toString() {
        return "DictionaryTranslator{" +
                "locale=" + locale +
                '}';
    }
}
