package ru.yandex.direct.i18n.localization;

import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.i18n.I18NException;
import ru.yandex.direct.i18n.Language;

import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.i18n.Language.EN;
import static ru.yandex.direct.i18n.Language.RU;
import static ru.yandex.direct.i18n.Language.TR;
import static ru.yandex.direct.i18n.Language.UK;


/**
 * Локализует объект по указанной локали
 *
 * @param <S> - тип локализируемого объекта
 * @param <L> - тип локализированного объекта
 */
@ParametersAreNonnullByDefault
public class LocalizationMapper<S, L> {

    private final Map<Language, List<Language>> translationChain = Map.of(
            RU, singletonList(RU),
            EN, asList(EN, RU),
            UK, asList(UK, RU, EN),
            TR, asList(TR, EN, RU)
    );


    private final BiConsumer<L, String> resultFieldSetter;
    private final Map<Language, Function<S, String>> sourceFieldGetters;
    private final Language defaultLanguage;
    private final Function<S, L> resultCreator;


    public static LocalizationMapperBuilder builder() {
        return new LocalizationMapperBuilder();
    }


    LocalizationMapper(
            BiConsumer<L, String> resultFieldSetter,
            Map<Language, Function<S, String>> sourceFieldGetters,
            Language defaultLanguage,
            Function<S, L> resultCreator
    ) {
        this.resultFieldSetter = resultFieldSetter;
        this.sourceFieldGetters = sourceFieldGetters;
        this.defaultLanguage = defaultLanguage;
        this.resultCreator = resultCreator;
    }


    /**
     * Локализация объекта
     *
     * @param source - исходный объект
     * @param locale - локаль, по которй строится перевод
     * @return Локализированный объект
     */
    public L localize(S source, Locale locale) {
        return localize(source, locale, null);
    }


    /**
     * Локализация объекта со значением по умолчанию
     *
     * @param source       - исходный объект
     * @param locale       - текущая локаль, по которй строится перевод
     * @param defaultValue - если не нашлось перевода, то значение по умолчанию
     * @return локализированный под текующую локаль объект
     */
    public L localize(
            S source,
            Locale locale,
            @Nullable String defaultValue
    ) {
        L result = resultCreator.apply(source);

        var sourceValue = getSourceValue(source, locale);
        resultFieldSetter.accept(
                result,
                sourceValue.orElse(defaultValue)
        );
        return result;
    }


    private Optional<String> getSourceValue(S source, Locale locale) {
        Language currentLanguage;
        try {
            currentLanguage = Language.fromLocale(locale);
        } catch (I18NException e) {
            currentLanguage = defaultLanguage;
        }

        Language translateTo = translationChain
                .get(currentLanguage)
                .stream()
                .filter(language -> isTranslationPresent(language, source))
                .findFirst()
                .orElse(null);

        return translateTo == null
                ? Optional.empty()
                : Optional.ofNullable(sourceFieldGetters.get(translateTo).apply(source));
    }


    private boolean isTranslationPresent(Language language, S source) {
        return sourceFieldGetters.containsKey(language)                     // есть геттер на поле для языка
                && sourceFieldGetters.get(language).apply(source) != null;  // само поле задано
    }

}
