package ru.yandex.direct.i18n.localization;

import java.util.HashMap;
import java.util.Map;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.function.Supplier;

import javax.annotation.Nonnull;

import ru.yandex.direct.i18n.Language;

import static com.google.common.base.Preconditions.checkNotNull;

public class LocalizationMapperBuilder {

    LocalizationMapperBuilder() {

    }


    public <S> FieldMapper<S> addRuTranslation(Function<S, String> getter) {
        return addToFieldMapper(Language.RU, getter);
    }

    public <S> FieldMapper<S> addEnTranslation(Function<S, String> getter) {
        return addToFieldMapper(Language.EN, getter);
    }

    public <S> FieldMapper<S> addTrTranslation(Function<S, String> getter) {
        return addToFieldMapper(Language.TR, getter);
    }

    public <S> FieldMapper<S> addUaTranslation(Function<S, String> getter) {
        return addToFieldMapper(Language.UK, getter);
    }

    private <S> FieldMapper<S> addToFieldMapper(Language language, Function<S, String> getter) {
        checkNotNull(getter, "Field getter function must be not null");
        return new FieldMapper<S>()
                .addTranslation(language, getter);
    }


    public final static class FieldMapper<S>  {

        private Map<Language, Function<S, String>> gettersMap = new HashMap<>();

        public FieldMapper<S> addRuTranslation(Function<S, String> getter) {
            return addTranslation(Language.RU, getter);
        }

        public FieldMapper<S> addEnTranslation(Function<S, String> getter) {
            return addTranslation(Language.EN, getter);
        }

        public FieldMapper<S> addTrTranslation(Function<S, String> getter) {
            return addTranslation(Language.TR, getter);
        }

        public FieldMapper<S> addUaTranslation(Function<S, String> getter) {
            return addTranslation(Language.UK, getter);
        }

        private FieldMapper<S> addTranslation(Language language, @Nonnull Function<S, String> getter) {
            gettersMap.put(language, getter);
            return this;
        }

        public <L> BuilderWithoutCreator<S, L> translateTo(BiConsumer<L, String> setter) {
            checkNotNull(setter, "Setter must be not null");
            return new BuilderWithoutCreator<>(gettersMap, setter);
        }

    }

    public static final class BuilderWithoutCreator<S, L> {
        private final BiConsumer<L, String> setter;
        private final Map<Language, Function<S, String>> getters;

        private BuilderWithoutCreator(Map<Language, Function<S, String>> getters, BiConsumer<L, String> setter) {
            this.getters = getters;
            this.setter = setter;
        }

        public AdditionalBuilderWithCreator<S, L> createBy(Supplier<L> supplier) {
            checkNotNull(supplier, "Result supplier must be not null");
            return new AdditionalBuilderWithCreator<>(getters, setter, (ignored) -> supplier.get());
        }

        public AdditionalBuilderWithCreator<S, L> copyBy(Function<S, L> copyCreator) {
            checkNotNull(copyCreator, "Copy function must be not null");
            return new AdditionalBuilderWithCreator<>(getters, setter, copyCreator);
        }
    }

    public static final class AdditionalBuilderWithCreator<S, L> {
        /**
         * Язык по умолчанию
         */
        private Language defaultLanguage = Language.EN;

        private final Map<Language, Function<S, String>> getters;
        private final BiConsumer<L, java.lang.String> setter;
        private Function<S, L> resultCreator;

        private AdditionalBuilderWithCreator(
                Map<Language, Function<S, String>> getters,
                BiConsumer<L, String> setter,
                Function<S, L> creator
        ) {
            this.getters = getters;
            this.setter = setter;
            this.resultCreator = creator;
        }

        public AdditionalBuilderWithCreator<S, L> withDefaultLanguage(Language language) {
            checkNotNull(language, "Default language must be not null");
            this.defaultLanguage = language;
            return this;
        }

        public LocalizationMapper<S, L> build() {
            return new LocalizationMapper<>(
                    setter,
                    getters,
                    defaultLanguage,
                    resultCreator
            );
        }
    }
}
