package ru.yandex.direct.i18n.types;

import java.text.MessageFormat;
import java.util.Arrays;

import ru.yandex.direct.i18n.Translatable;
import ru.yandex.direct.i18n.Translator;

/**
 * Переводимая строка, представляющая собой шаблон MessageFormat.
 * Чтобы понять как работает см. метод translate.
 */
public class MessageFormatTranslatable implements Translatable {
    private Translatable template;
    private Object[] args;

    public MessageFormatTranslatable(Translatable template, Object[] args) {
        this.template = template;
        this.args = args;
    }

    @Override
    public String translate(Translator translator) {
        return new MessageFormat(
                escapedPattern(translator),
                translator.getLocale()
        ).format(processArgs(args, translator));
    }

    private String escapedPattern(Translator translator) {
        String pattern = this.template.translate(translator);
        // Из доки на MessagePattern: A single quote itself must be represented by doubled single quotes '' throughout a String.
        pattern = pattern.replace("'", "''");
        return pattern;
    }

    private static Object[] processArgs(Object[] initialArgs, Translator translator) {
        if (initialArgs == null) {
            return null;
        } else {
            Object[] args = new Object[initialArgs.length];

            for (int i = 0; i < initialArgs.length; i++) {
                if (initialArgs[i] == null) {
                    args[i] = null;
                    continue;
                }
                Class argType = initialArgs[i].getClass();
                if (Translatable.class.isAssignableFrom(argType)) {
                    args[i] = ((Translatable) initialArgs[i]).translate(translator);
                } else {
                    args[i] = initialArgs[i];
                }
            }
            return args;
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        MessageFormatTranslatable that = (MessageFormatTranslatable) o;

        if (template != null ? !template.equals(that.template) : that.template != null) {
            return false;
        }
        // Probably incorrect - comparing Object[] arrays with Arrays.equals
        return Arrays.equals(args, that.args);

    }

    @Override
    public int hashCode() {
        int result = template != null ? template.hashCode() : 0;
        result = 31 * result + Arrays.hashCode(args);
        return result;
    }

    @Override
    public String toString() {
        return "MessageFormatTranslatable{" +
                "template=" + template +
                ", args=" + Arrays.toString(args) +
                '}';
    }
}
