package ru.yandex.direct.imagesearch;

import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.asynchttpclient.Param;
import org.asynchttpclient.util.Utf8UrlEncoder;

import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.http.smart.core.CallsPack;
import ru.yandex.direct.http.smart.core.Smart;
import ru.yandex.direct.imagesearch.model.ImageDoc;

import static ru.yandex.direct.http.smart.error.ErrorUtils.checkResultForErrors;
import static ru.yandex.direct.imagesearch.ImageSearchApi.SEARCH_TEXT_PARAM_NAME;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

public class ImageSearchClient {

    private static final int CHUNK_SIZE_FOR_SEARCH_BY_TEXT = 1; //по одному тексту на запрос

    private final ImageSearchApi imageSearchApi;

    public ImageSearchClient(String baseUrl, ParallelFetcherFactory fetcherFactory) {
        imageSearchApi = Smart.builder()
                .withParallelFetcherFactory(fetcherFactory)
                .withProfileName("image_search_client")
                .withBaseUrl(baseUrl)
                .build()
                .create(ImageSearchApi.class);
    }

    public List<ImageDoc> getImagesByUrl(String url, int imageLimit) {
        return getImagesByText("url:" + url, imageLimit);
    }

    public List<ImageDoc> getImagesByDomain(String domain, int imageLimit) {
        return getImagesByText("site:" + domain, imageLimit);
    }

    private List<ImageDoc> getImagesByText(String text, int imageLimit) {
        var calls = imageSearchApi.getImageByText(Set.of(text), ImageSearchApi.IMAGE_SIZE_ANY, imageLimit, 0, 0, 1);
        var results = calls.execute(CHUNK_SIZE_FOR_SEARCH_BY_TEXT).values().iterator().next();
        checkResultForErrors(results, ImageSearchClientException::new);
        return results.getSuccess();
    }

    public Map<String, List<ImageDoc>> getImagesByTextsAndUrl(Set<String> texts, String url, int imageLimit) {
        Map<String, String> textWithUrlToSourceText = StreamEx.of(texts)
                .toMap(
                        t -> t + " url:" + url,
                        t -> t
                );
        var calls = imageSearchApi.getImageByText(textWithUrlToSourceText.keySet(), ImageSearchApi.IMAGE_SIZE_ANY, imageLimit, 0, 0, 1);
        var results = calls.execute(CHUNK_SIZE_FOR_SEARCH_BY_TEXT);

        Map<String, List<ImageDoc>> data = resultsToMapOfResultByText(textWithUrlToSourceText.keySet(), calls, results);
        data = EntryStream.of(data)
                .mapKeys(textWithUrlToSourceText::get)
                .toMap();

        // Если хотя бы один запрос вернул непустой ответ, то ошибки остальных запросов можно не прокидывать вверх
        if (data.isEmpty()) {
            results.values().forEach(r -> checkResultForErrors(r, ImageSearchClientException::new));
        }

        return data;
    }

    public Map<String, List<ImageDoc>> getImagesByTextsAndDomain(Set<String> texts, String domain, int imageLimit) {
        var calls = imageSearchApi.getImageByTextAndDomain(texts, domain, ImageSearchApi.IMAGE_SIZE_ANY, imageLimit, 0, 0, 1);
        var results = calls.execute(CHUNK_SIZE_FOR_SEARCH_BY_TEXT);

        Map<String, List<ImageDoc>> data = resultsToMapOfResultByText(texts, calls, results);

        // Если хотя бы один запрос вернул непустой ответ, то ошибки остальных запросов можно не прокидывать вверх
        if (data.isEmpty()) {
            results.values().forEach(r -> checkResultForErrors(r, ImageSearchClientException::new));
        }

        return data;
    }

    private static Map<String, List<ImageDoc>> resultsToMapOfResultByText(Set<String> queryTexts, CallsPack<List<ImageDoc>> calls, Map<Long, Result<List<ImageDoc>>> results) {
        Map<String, String> encodeQueryToQuery = listToMap(queryTexts, Utf8UrlEncoder::encodeQueryElement);
        return StreamEx.of(calls.getRequests())
                .mapToEntry(
                        request -> StreamEx.of(request.getAHCRequest().getQueryParams())
                                .filter(param -> SEARCH_TEXT_PARAM_NAME.equals(param.getName()))
                                .map(Param::getValue)
                                .map(encodeQueryToQuery::get)
                                .nonNull()
                                .findFirst().orElse(null),
                        request -> results.get(request.getId())
                ) // map [text -> resultOfRequest]
                .nonNullKeys()
                .nonNullValues()
                .filterValues(result -> result.getErrors() == null)
                .mapValues(Result::getSuccess)
                .nonNullValues()
                .filterValues(images -> !images.isEmpty())
                .toMap();
    }
}
