package ru.yandex.direct.interruption;

import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * Позволяет потокам выставлять таймаут на выполнения некоторой работы внутри них.
 * Предназначен для использования в потоках трэд-пулов.
 * <p>
 * Перед началом работы, время которой требуется ограничить, вызывается метод start().
 * Сразу после окончания работы вызывается метод stop() в том же потоке!
 * <p>
 * Вызов метода start() шедулит джобу на прерывание текущего потока по истечении таймаута.
 * Вызов метода stop() отменяет прерывание текущего потока, если он еще не был прерван.
 * Гарантируется, что после вызова метода stop() поток не будет прерван. Однако, к моменту
 * вызова stop() на текущем потоке уже может быть вызван метод {@link Thread#interrupt()}.
 * <p>
 * См. {@link Thread#interrupt()}.
 * <p>
 * ВАЖНО: объект класса может и должен работать в единственном экземпляре для нескольких потоков.
 * ВАЖНО: после вызова start() всегда должен быть вызван stop() в том же потоке
 * для очистки ThreadLocal-переменных.
 * <p>
 * Пример использования:
 * <p>
 * <pre> {@code
 * public class Task implements Runnable {
 *
 *     private TimeoutInterrupter timeoutInterrupter;
 *
 *     @Override
 *     public void run() {
 *         timeoutInterrupter.start();
 *         try {
 *             doTheJob();
 *         } finally {
 *             timeoutInterrupter.stop();
 *         }
 *     }
 *
 *     private void doTheJob() {
 *         // main job
 *     }
 * }
 *
 * public void execute(Task task) {
 *     executor.execute(task);
 * }
 * }</pre>
 */
public class TimeoutInterrupter {

    private static final Logger logger = LoggerFactory.getLogger(TimeoutInterrupter.class);

    private final int timeout;
    private final ThreadLocal<InterrupterTask> interrupterTasks = new ThreadLocal<>();
    private final ScheduledThreadPoolExecutor executor;

    /**
     * @param timeout таймаут в секундах
     */
    public TimeoutInterrupter(int timeout) {
        checkArgument(timeout > 0, "timeout must be greater than zero");
        this.timeout = timeout;
        this.executor = new ScheduledThreadPoolExecutor(1);
        this.executor.setRemoveOnCancelPolicy(true);
        ThreadFactory threadFactory = new ThreadFactoryBuilder()
                .setNameFormat("TimeoutInterrupter-%d")
                .setDaemon(true)
                .build();
        this.executor.setThreadFactory(threadFactory);
    }

    /**
     * Зашедулить задачу на прерывание текущего потока через установленный таймаут
     */
    public void start() {
        forcePreviousIterationStop();
        Interrupter interrupter = new Interrupter(Thread.currentThread());
        ScheduledFuture<?> futureTask = executor.schedule(interrupter, timeout, TimeUnit.SECONDS);
        interrupterTasks.set(new InterrupterTask(interrupter, futureTask));
    }

    /**
     * Отменить задачу на прерывание текущего потока
     */
    public void stop() {
        InterrupterTask interrupterTask = interrupterTasks.get();
        if (interrupterTask != null) {
            interrupterTask.interrupter.cancel();
            interrupterTask.futureTask.cancel(false);
            interrupterTasks.remove();
        }
    }

    /**
     * Закрыть шедулер и его трэд-пул.
     */
    public void close() {
        executor.shutdownNow();
    }

    private void forcePreviousIterationStop() {
        InterrupterTask interrupterTask = interrupterTasks.get();
        if (interrupterTask != null) {
            logger.error("start() called before stop() in previous iteration. "
                    + "Previous iteration will be forced to stop"
                    + "(it can be actually completed now or not).");
            stop();
        }
    }


    /**
     * Задача на прерывание потока
     */
    private static class Interrupter implements Runnable {

        private final Thread threadToInterrupt;
        private volatile boolean toInterrupt = true;

        public Interrupter(Thread threadToInterrupt) {
            this.threadToInterrupt = threadToInterrupt;
        }

        @Override
        public synchronized void run() {
            if (toInterrupt) {
                logger.warn("Timeout expired, interrupt thread...");
                threadToInterrupt.interrupt();
                toInterrupt = false;
            }
        }

        public synchronized void cancel() {
            if (toInterrupt) {
                toInterrupt = false;
                logger.trace("Timeout cancelled");
            } else {
                logger.trace("Can not cancel timeout after Thread.interrupt() invoked");
            }
        }
    }

    /**
     * Контейнер для удобного хранения данных в ThreadLocal
     */
    private static class InterrupterTask {

        private final Interrupter interrupter;
        private final ScheduledFuture<?> futureTask;

        public InterrupterTask(Interrupter interrupter, ScheduledFuture<?> futureTask) {
            this.interrupter = interrupter;
            this.futureTask = futureTask;
        }
    }
}
