package ru.yandex.direct.inventori;

import java.io.IOException;
import java.util.UUID;
import java.util.concurrent.ExecutionException;

import javax.annotation.PreDestroy;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import org.asynchttpclient.DefaultAsyncHttpClient;
import org.asynchttpclient.DefaultAsyncHttpClientConfig;
import org.asynchttpclient.Request;
import org.asynchttpclient.RequestBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class InventoriCampaignsPredictionClient {

    private static final Logger logger = LoggerFactory.getLogger(InventoriCampaignsPredictionClient.class);
    private final String inventoryUrl;
    private final DefaultAsyncHttpClient client;
    private final ObjectReader reader;

    public InventoriCampaignsPredictionClient(String inventoryUrl) {
        this.inventoryUrl = inventoryUrl;
        logger.info("inventoryUrl: " + inventoryUrl);
        // Возможно здесь лучше воспользоваться общим клиентом, который у нас есть в CommonConfig'e.
        // Клиент, созданный таким образом, не будет автоматически закрыт, и его non-daemon тред будет висеть,
        // не давая приложению завершиться (чтобы это пофиксить был добавлен метод с анотацией @PreDestroy ниже)
        this.client = new DefaultAsyncHttpClient(new DefaultAsyncHttpClientConfig.Builder().build());
        this.reader = new ObjectMapper().readerFor(new TypeReference<InventoriGeneralCampaignsPredictionResponse>() {});
    }

    /**
     * Если кампания влазит по данным inventory и её можно подтвердить возвращает TRAFFIC_LIGHT_GREEN
     * Если кампания не влазит по данным inventory возвращает TRAFFIC_LIGHT_RED
     *
     * @param campaignId cid
     * @return десериализованный ответ inventory
     * @throws InterruptedException если операция прервана
     * @throws RuntimeException если запрос в инвентори закончился неудачей или не получилось десериализовать ответ
     */
    public InventoriGeneralCampaignsPredictionResponse generalCampaignsPrediction(
            Long campaignId) throws InterruptedException {
        String stringResponse;
        try {
            stringResponse = getStringResponse(campaignId.toString());
        } catch (ExecutionException e) {
            throw new RuntimeException("Error while executing inventory request", e);
        }
        try {
            return reader.readValue(stringResponse);
        } catch (IOException e) {
            throw new RuntimeException("Error while deserializing value " + stringResponse, e);
        }
    }

    protected String getStringResponse(String campaignId) throws ExecutionException, InterruptedException {
        final RequestBuilder builder = new RequestBuilder()
                .setUrl(inventoryUrl)
                .addQueryParam("cid", campaignId)
                .addHeader("X-Request-ID", generateRequestId())
                .addHeader("X-Yandex-Direct-CampaignId", campaignId)
                .addHeader("X-Yandex-Direct-OperatorLogin", "booking_auto_allocation");
        final Request request = builder.build();
        logger.info("Inventory request: " + request.toString());
        return client.executeRequest(request).get().getResponseBody();
    }

    private static String generateRequestId() {
        return UUID.randomUUID().toString().toUpperCase();
    }

    @PreDestroy
    public void close() {
        client.close();
    }
}
