package ru.yandex.direct.inventori;

import java.nio.charset.Charset;
import java.time.Duration;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.netty.handler.codec.http.HttpHeaderNames;
import io.netty.handler.codec.http.HttpHeaderValues;
import io.netty.handler.codec.http.HttpMethod;
import io.netty.handler.codec.http.HttpResponseStatus;
import one.util.streamex.StreamEx;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.Param;
import org.asynchttpclient.Request;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.asynchttp.AsyncHttpExecuteException;
import ru.yandex.direct.asynchttp.ErrorResponseWrapperException;
import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.JsonParsableRequest;
import ru.yandex.direct.asynchttp.ParallelFetcher;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.ParsableRequest;
import ru.yandex.direct.inventori.model.request.CampaignParameters;
import ru.yandex.direct.inventori.model.request.CampaignPredictionRequest;
import ru.yandex.direct.inventori.model.request.ForecastRequest;
import ru.yandex.direct.inventori.model.request.Target;
import ru.yandex.direct.inventori.model.response.CampaignPredictionResponse;
import ru.yandex.direct.inventori.model.response.ForecastResponse;
import ru.yandex.direct.inventori.model.response.GeneralForecastResponse;
import ru.yandex.direct.inventori.model.response.IndoorPredictionResponse;
import ru.yandex.direct.inventori.model.response.OutdoorPredictionResponse;
import ru.yandex.direct.solomon.SolomonUtils;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceChild;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.JsonUtils.fromJson;
import static ru.yandex.direct.utils.net.YandexHttpHeaders.DIRECT_SOURCE;
import static ru.yandex.direct.utils.net.YandexHttpHeaders.X_REQUEST_ID;
import static ru.yandex.direct.utils.net.YandexHttpHeaders.X_YANDEX_DIRECT_ADGROUP_ID;
import static ru.yandex.direct.utils.net.YandexHttpHeaders.X_YANDEX_DIRECT_CAMPAIGN_ID;
import static ru.yandex.direct.utils.net.YandexHttpHeaders.X_YANDEX_DIRECT_CLIENT_LOGIN;
import static ru.yandex.direct.utils.net.YandexHttpHeaders.X_YANDEX_DIRECT_OPERATOR_LOGIN;
import static ru.yandex.direct.utils.net.YandexHttpHeaders.X_YANDEX_SOURCE_ID;

@ParametersAreNonnullByDefault
public class InventoriClient {
    private static final Logger logger = LoggerFactory.getLogger(InventoriClient.class);

    private static final String BUDGET_PERFORMANCE_API_PATH = "budget_performance";
    private static final String TRAFFIC_LIGHT_PREDICTION_API_PATH = "traffic_light_prediction";
    private static final String FORECAST_API_PATH = "group_reach_prediction";
    private static final String GENERAL_GROUP_REACH_PREDICTION_API_PATH = "general_group_reach_prediction";
    private static final String GENERAL_CAMPAIGN_PREDICTION_API_PATH = "general_campaign_prediction";
    private static final String GENERAL_RECOMMENDATION_API_PATH = "general_recommendation";
    private static final String PARAMETRISED_CAMPAIGN_PREDICTION_API_PATH = "parametrised_campaign_prediction";

    public static final Long DEFAULT_REACH_LESS_THAN = 1000L;

    private static final ObjectMapper mapper = new ObjectMapper();
    public static final String SERVICE_NAME = "inventori.client";

    private final InventoriClientConfig config;
    private final String inventoriServiceBaseUrl;
    private final AsyncHttpClient asyncHttpClient;
    private final ParallelFetcherFactory fetcherFactory;
    private final Duration requestTimeout;

    public InventoriClient(AsyncHttpClient asyncHttpClient,
                           InventoriClientConfig config) {
        this.config = config;
        this.requestTimeout = config.getRequestTimeout();
        this.asyncHttpClient = asyncHttpClient;
        this.inventoriServiceBaseUrl = config.getBaseUrl();
        this.fetcherFactory = new ParallelFetcherFactory(asyncHttpClient, new FetcherSettings()
                .withRequestRetries(config.getRequestRetries())
                .withRequestTimeout(config.getRequestTimeout())
                .withParallel(config.getParallel()));
    }

    /**
     * Прогноз недельного охвата рекламной кампании в Директе
     *
     * @param requestCorrelationId уникальный ID группы запросов для удобства поиска в логах
     * @param target               запрос с таргетами для прогнозатора
     * @param campaignId           ID кампании
     * @param adgroupId            ID группы объявлений
     * @param clientLogin          логин пользователя
     * @param operatorLogin        логин оператора
     * @return {@link ForecastResponse}
     * @see <a href="https://wiki.yandex-team.ru/InventORIInDirect/API/#/api/forecastpost">InventORI In Direct</a>
     */
    public ForecastResponse getForecast(String requestCorrelationId, Target target, @Nullable Long campaignId,
                                        @Nullable Long adgroupId, String clientLogin, @Nullable String operatorLogin) {
        return requestService(new ForecastRequest(target), requestCorrelationId, campaignId, adgroupId,
                clientLogin, operatorLogin, FORECAST_API_PATH, ForecastResponse.class);
    }

    /**
     * Прогноз прогноз недельных охватов (общий и уточненный) группы в мастере кампаний
     *
     * @param requestCorrelationId уникальный ID группы запросов для удобства поиска в логах
     * @param target               запрос с таргетами для прогнозатора
     * @param campaignId           ID кампании
     * @param adgroupId            ID группы объявлений
     * @param clientLogin          логин пользователя
     * @param operatorLogin        логин оператора
     * @return {@link GeneralForecastResponse}
     * @see <a href="https://wiki.yandex-team.ru/inventori/api/general_group_reach_prediction/">InventORI In Direct</a>
     */
    public GeneralForecastResponse getGeneralForecast(String requestCorrelationId, Target target, @Nullable Long campaignId,
                                        @Nullable Long adgroupId, String clientLogin, @Nullable String operatorLogin) {
        return requestService(new ForecastRequest(target), requestCorrelationId, campaignId, adgroupId,
                clientLogin, operatorLogin, GENERAL_GROUP_REACH_PREDICTION_API_PATH, GeneralForecastResponse.class);
    }

    /**
     * Прогноз недельных охвата и емкости OTS для OUTDOOR
     *
     * @param requestCorrelationId уникальный ID группы запросов для удобства поиска в логах
     * @param target               запрос с таргетами для прогнозатора
     * @param campaignId           ID кампании
     * @param adgroupId            ID группы объявлений
     * @param clientLogin          логин пользователя
     * @param operatorLogin        логин оператора
     * @return {@link OutdoorPredictionResponse}
     * @see
     * <a href="https://wiki.yandex-team.ru/InventORIInDirect/API/#2/api/outdoorprediction/reachandotscapacitypredictionpost-prognoznedelnyxoxvataiemkostiotsdljaoutdoor">InventORI In Direct</a>
     */
    public OutdoorPredictionResponse getOutdoorPrediction(String requestCorrelationId, Target target,
                                                          @Nullable Long campaignId, @Nullable Long adgroupId,
                                                          String clientLogin, @Nullable String operatorLogin) {
        // Заглушка, чтобы не ходили в инвентори
        // Наружняя реклама отрывается в тикете https://st.yandex-team.ru/DIRECT-127677
        return new OutdoorPredictionResponse().withReachLessThan(DEFAULT_REACH_LESS_THAN);

    }

    /**
     * Прогноз недельных охвата и емкости OTS для INDOOR
     *
     * @param requestCorrelationId уникальный ID группы запросов для удобства поиска в логах
     * @param target               запрос с таргетами для прогнозатора
     * @param campaignId           ID кампании
     * @param adgroupId            ID группы объявлений
     * @param clientLogin          логин пользователя
     * @param operatorLogin        логин оператора
     * @return {@link IndoorPredictionResponse}
     * @see
     * <a href="https://wiki.yandex-team.ru/InventORIInDirect/API/#3/api/indoorprediction/otscapacitypredictionpost-prognoznedelnojjemkostiotsdljaindoor">InventORI In Direct</a>
     */
    public IndoorPredictionResponse getIndoorPrediction(String requestCorrelationId, Target target,
                                                        @Nullable Long campaignId, @Nullable Long adgroupId,
                                                        String clientLogin,
                                                        @Nullable String operatorLogin) {
        // Заглушка, чтобы не ходили в инвентори
        // Внутренняя реклама отрывается в тикете https://st.yandex-team.ru/DIRECT-127677
        return new IndoorPredictionResponse().withReachLessThan(DEFAULT_REACH_LESS_THAN);
    }

    /**
     * Прогноз охвата для кампании по всем группам
     *
     * @param requestId     ID запроса
     * @param operatorLogin Логин оператора
     * @param clientLogin   Логин клиента
     * @param request       Тело запроса
     */
    public CampaignPredictionResponse getGeneralCampaignPrediction(
            String requestId,
            String operatorLogin,
            String clientLogin,
            CampaignPredictionRequest request) throws JsonProcessingException {
        return getCampaignPrediction(requestId, operatorLogin, clientLogin, request,
                GENERAL_CAMPAIGN_PREDICTION_API_PATH, null);

//        return requestService(request, requestId, request.getCampaignId(), null,
//                clientLogin, operatorLogin, GENERAL_CAMPAIGN_PREDICTION_API_PATH,
//                GeneralCampaignPredictionResponse.class);
    }

    /**
     * Прогноз охвата для кампании по всем группам
     * @param requestId     ID запроса
     * @param operatorLogin Логин оператора
     * @param clientLogin   Логин клиента
     * @param request       Тело запроса
     * @return
     */
    public CampaignPredictionResponse getGeneralRecommendation(
            String requestId,
            String operatorLogin,
            String clientLogin,
            CampaignPredictionRequest request) throws JsonProcessingException
    {
        return getCampaignPrediction(requestId, operatorLogin, clientLogin, request, GENERAL_RECOMMENDATION_API_PATH, null);
    }

    /**
     * Отправка запроса в Inventori сервис с помощью {@link AsyncHttpClient}
     *
     * @param body                 тело запроса
     * @param requestCorrelationId уникальный ID группы запросов для удобства поиска в логах (используется для отладки)
     * @param campaignId           ID кампании (используется для отладки)
     * @param adgroupId            ID группы объявлений (используется для отладки)
     * @param clientLogin          логин пользователя (используется для отладки)
     * @param operatorLogin        логин оператора (используется для отладки)
     * @param apiPath              путь к ручке прогнозатора
     * @param responseClass        класс, в который надо сконвертировать ответ ручки
     * @param <T>                  класс, в который надо сконвертировать ответ ручки
     * @return ответ сервиса inventori
     */
    private <T> T requestService(Object body, String requestCorrelationId, @Nullable Long campaignId,
                                 @Nullable Long adgroupId, String clientLogin, @Nullable String operatorLogin,
                                 String apiPath,
                                 Class<T> responseClass) {
        Response result;
        try (TraceProfile profile = Trace.current().profile(apiPath)) {

            final String bodyJson = JsonUtils.toJson(body);
            final RequestBuilder builder = new RequestBuilder("POST")
                    .setUrl(inventoriServiceBaseUrl + apiPath)
                    .setCharset(Charset.forName("UTF-8"))
                    .setBody(bodyJson)
                    .addHeader(HttpHeaderNames.ACCEPT, HttpHeaderValues.APPLICATION_JSON)
                    .addHeader(HttpHeaderNames.CONTENT_TYPE, HttpHeaderValues.APPLICATION_JSON)
                    // уникальный ID запроса, формируется на стороне клиента (Direct)
                    .addHeader(X_REQUEST_ID, requestCorrelationId)
                    .addHeader(X_YANDEX_DIRECT_CAMPAIGN_ID,
                            Optional.ofNullable(campaignId).map(String::valueOf).orElse(null))
                    .addHeader(X_YANDEX_DIRECT_ADGROUP_ID,
                            Optional.ofNullable(adgroupId).map(String::valueOf).orElse(null))
                    .addHeader(X_YANDEX_DIRECT_OPERATOR_LOGIN, operatorLogin)
                    .addHeader(X_YANDEX_DIRECT_CLIENT_LOGIN, clientLogin)
                    .setHeader(X_YANDEX_SOURCE_ID, DIRECT_SOURCE);
            final Request request = builder.build();

            logger.info("Request: {}\t{}", request, bodyJson);
            result = asyncHttpClient.executeRequest(request).get(requestTimeout.toMillis(), TimeUnit.MILLISECONDS);
            if (result.getHeader(X_REQUEST_ID) == null) {
                result.getHeaders().add(X_REQUEST_ID, requestCorrelationId);
            }
            logger.info("Response: {}", result);

            if (result.getStatusCode() >= HttpResponseStatus.INTERNAL_SERVER_ERROR.code()) {
                throw new InventoriException(requestCorrelationId, "Got error on response for Inventori request "
                        + request + ", result:" + result
                );
            }
        } catch (TimeoutException ex) {
            logger.error("Request was timed out. X-Request-ID={}", requestCorrelationId);
            throw new InventoriException(requestCorrelationId, ex);
        } catch (ExecutionException | InterruptedException ex) {
            logger.error("Request was unexpectedly interrupted. X-Request-ID={}", requestCorrelationId);
            throw new InventoriException(requestCorrelationId, ex);
        }

        final T response = fromJson(result.getResponseBody(), responseClass);
        if (response == null) {
            throw new InventoriException(requestCorrelationId,
                    "Deserialization result is null. X-Request-ID=" + requestCorrelationId);
        }

        return response;
    }

    /**
     * Выполняет запрос к CPM-прогнозатору.
     *
     * @param requestId     ID запроса
     * @param operatorLogin Логин оператора
     * @param clientLogin   Логин клиента
     * @param request       Тело запроса
     * @param apiPath       Путь к прогнозатору
     * @param params        Параметры
     */
    private CampaignPredictionResponse getCampaignPrediction(String requestId, String operatorLogin, String clientLogin,
                                                             CampaignPredictionRequest request, String apiPath,
                                                             @Nullable List<Param> params) throws JsonProcessingException {
        try (TraceProfile profile = Trace.current().profile(apiPath)) {
            TraceChild traceChild = Trace.current().child(SERVICE_NAME, apiPath);

            var corrections = nvl(request.getParameters(), new CampaignParameters()).getCorrections();
            if (corrections != null) {
                // Добавляем в запросе на прогноз корректировку inbanner если присутствует inapp (videoInterstitial)
                var trafficTypeCorrection = corrections.getTrafficTypeCorrections();
                trafficTypeCorrection.setVideoInbanner(trafficTypeCorrection.getVideoInterstitial());
            }
            String body = mapper.writerFor(request.getClass()).writeValueAsString(request);

            Request req = prepareBudgetForecastBuilder(requestId, operatorLogin, clientLogin, request, apiPath)
                    .setQueryParams(params)
                    .setBody(body)
                    .build();
            ParsableRequest<CampaignPredictionResponse> parsableRequest = new JsonParsableRequest<>(
                    traceChild.getSpanId(),
                    req,
                    CampaignPredictionResponse.class);
            logger.info("Request: {}\t{}", req, body);
            try {
                MetricRegistry metricRegistry =
                        SolomonUtils.getParallelFetcherMetricRegistry(profile.getFunc());
                ParallelFetcher<CampaignPredictionResponse> parallelFetcher = fetcherFactory
                        .getParallelFetcherWithMetricRegistry(metricRegistry);
                return parallelFetcher.executeWithErrorsProcessing(parsableRequest).getSuccess();
            } catch (AsyncHttpExecuteException ex) {
                return StreamEx.of(ex.getSuppressed())
                        .select(ErrorResponseWrapperException.class)
                        .filter(exi -> exi.getResponse() != null)
                        .findFirst()
                        .map(r -> parsableRequest.getParseFunction().apply(r.getResponse()))
                        .orElseThrow(() -> ex);
            }
        }
    }

    /**
     * Выполняет запрос к CPM-прогнозатору для получения прогноза по нескольким бюджетам.
     *
     * @param requestId     ID запроса
     * @param operatorLogin Логин оператора
     * @param clientLogin   Логин клиента
     * @param request       Тело запроса
     * @param params        Параметры
     */
    public CampaignPredictionResponse getParametrisedCampaignPrediction(String requestId, String operatorLogin,
                                                                        String clientLogin, CampaignPredictionRequest request,
                                                                        @Nullable List<Param> params) throws JsonProcessingException {
        return getCampaignPrediction(requestId, operatorLogin, clientLogin, request,
                PARAMETRISED_CAMPAIGN_PREDICTION_API_PATH, params);
    }

    /**
     * Выполняет запрос к CPM-прогнозатору для получения прогноза по кампании.
     *
     * @param requestId     ID запроса
     * @param operatorLogin Логин оператора
     * @param clientLogin   Логин клиента
     * @param request       Тело запроса
     */
    public CampaignPredictionResponse getCampaignPrediction(String requestId, String operatorLogin, String clientLogin,
                                                            CampaignPredictionRequest request) throws JsonProcessingException {
        return getCampaignPrediction(requestId, operatorLogin, clientLogin, request, BUDGET_PERFORMANCE_API_PATH, null);
    }

    /**
     * Выполняет запрос к CPM-прогнозатору для получения цвета светофора и рекомендуемого cpm.
     *
     * @param requestId     ID запроса
     * @param operatorLogin Логин оператора
     * @param clientLogin   Логин клиента
     * @param request       Тело запроса
     */
    public CampaignPredictionResponse getTrafficLightPrediction(String requestId, String operatorLogin,
                                                                String clientLogin, CampaignPredictionRequest request) throws JsonProcessingException {
        return getCampaignPrediction(requestId, operatorLogin, clientLogin, request, TRAFFIC_LIGHT_PREDICTION_API_PATH, null);
    }

    /**
     * Проставляет URL и необходимые заголовки запроса
     *
     * @param requestId     ID запроса
     * @param operatorLogin Логин оператора
     * @param clientLogin   Логин клиента
     * @param request       Запрос
     * @param apiPath       Путь к прогнозатору
     */
    private RequestBuilder prepareBudgetForecastBuilder(String requestId, String operatorLogin, String clientLogin,
                                                        CampaignPredictionRequest request, String apiPath) {
        RequestBuilder builder = new RequestBuilder();

        if (request.getCampaignId() != null) {
            builder.addHeader(X_YANDEX_DIRECT_CAMPAIGN_ID, request.getCampaignId().toString());
        }

        return builder
                .setMethod(HttpMethod.POST.name())
                .setUrl(config.getBaseUrl() + apiPath)
                .addHeader(HttpHeaderNames.CONTENT_TYPE, HttpHeaderValues.APPLICATION_JSON)
                .addHeader(X_REQUEST_ID, requestId)
                .addHeader(X_YANDEX_DIRECT_OPERATOR_LOGIN, operatorLogin)
                .addHeader(X_YANDEX_DIRECT_CLIENT_LOGIN, clientLogin)
                .addHeader(X_YANDEX_SOURCE_ID, DIRECT_SOURCE);
    }
}
