package ru.yandex.direct.inventori.model.request;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.collect.ImmutableMap;
import org.jetbrains.annotations.NotNull;

import static java.util.Objects.requireNonNull;
import static ru.yandex.direct.inventori.model.request.CampaignParametersSchedule.StrategyField.AUTO_PROLONGATION;
import static ru.yandex.direct.inventori.model.request.CampaignParametersSchedule.StrategyField.IS_BOOKING;
import static ru.yandex.direct.inventori.model.request.StrategyType.FIX_PRICE;
import static ru.yandex.direct.inventori.model.request.StrategyType.MAX_AVG_CPV;
import static ru.yandex.direct.inventori.model.request.StrategyType.MAX_AVG_CPV_CUSTOM_PERIOD;
import static ru.yandex.direct.inventori.model.request.StrategyType.MAX_REACH;
import static ru.yandex.direct.inventori.model.request.StrategyType.MAX_REACH_CUSTOM_PERIOD;
import static ru.yandex.direct.inventori.model.request.StrategyType.MIN_CPM;
import static ru.yandex.direct.inventori.model.request.StrategyType.MIN_CPM_CUSTOM_PERIOD;

/**
 * Класс, реализующий тип объекта group (ранее schedule) в Инвентори
 * Описание полей: https://wiki.yandex-team.ru/inventoriindirect/api/campaign/campaignparameters/strategy/
 */
@ParametersAreNonnullByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
public class CampaignParametersSchedule {
    private static final DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");

    @JsonProperty("strategy_type")
    private StrategyType strategyType;

    @JsonProperty("budget")
    private long budget;

    @JsonProperty("start_date")
    private String startDate;

    @JsonProperty("end_date")
    private String endDate;

    @JsonProperty("max_avg_cpm")
    private Long cpm;

    @JsonProperty("max_avg_cpv")
    private Long cpv;

    @Nullable
    @JsonProperty("auto_prolongation")
    private Boolean autoProlongation;

    @Nullable
    @JsonProperty("is_booking")
    private Boolean isBooking;

    public CampaignParametersSchedule() {
    }

    private CampaignParametersSchedule(@NotNull StrategyType strategyType,
            long budget,
            @NotNull String startDate,
            @NotNull String endDate,
            @Nullable Long cpm,
            @Nullable Long cpv,
            @Nullable Boolean autoProlongation,
            @Nullable Boolean isBooking) {
        this.strategyType = requireNonNull(strategyType, "Strategy type is null");
        this.budget = budget;
        this.startDate = requireNonNull(LocalDate.parse(startDate), "End date is null").format(formatter);
        this.endDate = requireNonNull(LocalDate.parse(endDate), "End date is null").format(formatter);
        this.cpm = cpm;
        this.cpv = cpv;
        this.autoProlongation = autoProlongation;
        this.isBooking = isBooking;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        CampaignParametersSchedule that = (CampaignParametersSchedule) o;
        return budget == that.budget &&
                Objects.equals(strategyType, that.strategyType) &&
                Objects.equals(startDate, that.startDate) &&
                Objects.equals(endDate, that.endDate) &&
                Objects.equals(cpm, that.cpm) &&
                Objects.equals(cpv, that.cpv) &&
                Objects.equals(autoProlongation, that.autoProlongation) &&
                Objects.equals(isBooking, that.isBooking);
    }

    @Override
    public int hashCode() {
        return Objects.hash(strategyType, budget, startDate, endDate, cpm, cpv, autoProlongation, isBooking);
    }

    public static Builder builder() {
        return new Builder();
    }

    public static final class Builder {
        private StrategyType strategyType;
        private long budget;
        private String startDate;
        private String endDate;
        private Long cpm;
        private Long cpv;
        private Boolean autoProlongation;
        private Boolean isBooking;

        private Builder() {
        }

        public Builder withStrategyType(StrategyType strategyType) {
            this.strategyType = strategyType;
            return this;
        }

        public Builder withBudget(long budget) {
            this.budget = budget;
            return this;
        }

        public Builder withStartDate(String startDate) {
            this.startDate = startDate;
            return this;
        }

        public Builder withEndDate(String endDate) {
            this.endDate = endDate;
            return this;
        }

        public Builder withStartDate(LocalDate startDate) {
            this.startDate = startDate.format(formatter);
            return this;
        }

        public Builder withEndDate(LocalDate endDate) {
            this.endDate = endDate.format(formatter);
            return this;
        }

        public Builder withCpm(Long cpm) {
            this.cpm = cpm;
            return this;
        }

        public Builder withCpv(Long cpv) {
            this.cpv = cpv;
            return this;
        }

        public Builder withAutoProlongation(@Nullable Boolean autoProlongation) {
            this.autoProlongation = autoProlongation;
            return this;
        }

        public Builder withIsBooking(@Nullable Boolean isBooking) {
            this.isBooking = isBooking;
            return this;
        }

        public CampaignParametersSchedule build() {
            Set<StrategyField> fields = NULLABLE_FIELDS_BY_STRATEGY_TYPES.get(strategyType);
            return new CampaignParametersSchedule(
                    strategyType,
                    budget,
                    startDate,
                    endDate,
                    cpm,
                    cpv,
                    fields.contains(AUTO_PROLONGATION) ? autoProlongation : null,
                    fields.contains(IS_BOOKING) ? isBooking : null);
        }
    }
    private static final Map<StrategyType, Set<StrategyField>> NULLABLE_FIELDS_BY_STRATEGY_TYPES =
            ImmutableMap.<StrategyType, Set<StrategyField>>builder()
            .put(MAX_REACH,
                    Set.of( AUTO_PROLONGATION))
            .put(MAX_REACH_CUSTOM_PERIOD,
                    Set.of( AUTO_PROLONGATION))
            .put(MIN_CPM,
                    Set.of( AUTO_PROLONGATION))
            .put(MIN_CPM_CUSTOM_PERIOD,
                    Set.of( AUTO_PROLONGATION))
            .put(MAX_AVG_CPV,
                    Set.of( AUTO_PROLONGATION))
            .put(MAX_AVG_CPV_CUSTOM_PERIOD,
                    Set.of( AUTO_PROLONGATION))
            .put(FIX_PRICE,
                    Set.of( IS_BOOKING))
            .build();

    protected enum StrategyField {
        START_DATE,
        END_DATE,
        BUDGET,
        MAX_AVG_CPM,
        MAX_AVG_CPV,
        AUTO_PROLONGATION,
        IS_BOOKING
    }
}
