package ru.yandex.direct.inventori.model.request;

import java.util.List;
import java.util.Objects;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.collect.ImmutableList;

import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

@ParametersAreNonnullByDefault
@JsonInclude(JsonInclude.Include.NON_NULL)
public class CampaignPredictionRequest {

    @JsonProperty("cid")
    protected Long campaignId;

    @JsonProperty("default_target_id")
    protected Integer defaultTargetId;

    @JsonProperty("campaign_type")
    protected InventoriCampaignType campaignType;

    @JsonProperty("groups")
    protected List<Target> targets;

    @JsonProperty("campaign_parameters")
    protected CampaignParameters parameters;

    @JsonProperty("is_social")
    protected Boolean isSocial;

    @JsonProperty("has_brandlift")
    protected Boolean hasBrandlift;

    protected Boolean containsKeywordGroup;

    public CampaignPredictionRequest() {
    }

    /**
     * Создает запрос на оценку покрытия cpm-кампании
     *
     * @param campaignId           Идентифкатор существующей кампании, если null - должен быть указан тип примера для
     *                             новой кампании
     * @param defaultTargetId      Тип примера для новой кампании (0 - широкий охват, 1 - узкий интерес), если null -
     *                             должен быть указан идентификатор существующей кампании
     * @param targets              список данных по группам объявлений, null, если не указан campaginId
     * @param parameters           параметры кампании
     * @param containsKeywordGroup содержат ли группы объявлений группы на ключевых словах
     * @param isSocial             признак того, что кампания является социалкой
     */
    protected CampaignPredictionRequest(@Nullable Long campaignId, @Nullable Integer defaultTargetId,
                                     InventoriCampaignType campaignType,
                                     @Nullable List<Target> targets,
                                     CampaignParameters parameters,
                                     @Nullable Boolean containsKeywordGroup,
                                     @Nullable Boolean isSocial,
                                     @Nullable Boolean hasBrandlift) {
        this.campaignId = campaignId;
        this.defaultTargetId = defaultTargetId;
        this.campaignType = campaignType;
        this.targets = ifNotNull(targets, ImmutableList::copyOf);
        this.parameters = parameters;
        this.containsKeywordGroup = containsKeywordGroup;
        this.isSocial = isSocial;
        this.hasBrandlift = hasBrandlift;
    }

    @Nullable
    public Long getCampaignId() {
        return campaignId;
    }

    public Integer getDefaultTargetId() {
        return defaultTargetId;
    }

    public InventoriCampaignType getCampaignType() {
        return campaignType;
    }

    public List<Target> getTargets() {
        return targets;
    }

    public Boolean getIsSocial() {
        return isSocial;
    }

    public CampaignPredictionRequest setTargets(List<Target> targets) {
        this.targets = targets;
        return this;
    }

    public CampaignParameters getParameters() {
        return parameters;
    }

    public void setParameters(CampaignParameters parameters) {
        this.parameters = parameters;
    }

    public void setDefaultTargetId(Integer defaultTargetId) {
        this.defaultTargetId = defaultTargetId;
    }

    public void setCampaignType(InventoriCampaignType campaignType) {
        this.campaignType = campaignType;
    }

    public void setSocial(Boolean isSocial) {
        this.isSocial = isSocial;
    }

    public Boolean getHasBrandlift() {
        return hasBrandlift;
    }

    public void setHasBrandlift(Boolean hasBrandlift) {
        this.hasBrandlift = hasBrandlift;
    }

    public boolean containsKeywordGroup() {
        if (containsKeywordGroup == null) {
            return false;
        }
        return containsKeywordGroup;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        CampaignPredictionRequest request = (CampaignPredictionRequest) o;
        return Objects.equals(targets, request.targets) &&
                Objects.equals(parameters, request.parameters);
    }

    @Override
    public int hashCode() {
        return Objects.hash(targets, parameters);
    }

    @Override
    public String toString() {
        return "CampaignPredictionRequest{" +
                "campaignId=" + campaignId +
                ", defaultTargetId=" + defaultTargetId +
                ", campaignType=" + campaignType +
                ", targets=" + targets +
                ", parameters=" + parameters +
                ", isSocial=" + isSocial +
                ", hasBrandlift=" + hasBrandlift +
                ", containsKeywordGroup=" + containsKeywordGroup +
                '}';
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        protected Long campaignId;
        protected Integer defaultTargetId;
        protected InventoriCampaignType campaignType;
        protected List<Target> targets;
        protected CampaignParameters parameters;
        protected Boolean containsKeywordGroup;
        protected Boolean isSocial;
        protected Boolean hasBrandlift;

        protected Builder() {
        }

        public Builder withBrandlift(Boolean hasBrandlift) {
            this.hasBrandlift = hasBrandlift;
            return this;
        }

        public Builder withCampaignId(long campaignId) {
            this.campaignId = campaignId;
            return this;
        }

        public Builder withDefaultTargetId(Integer defaultTargetId) {
            this.defaultTargetId = defaultTargetId;
            return this;
        }

        public Builder withCampaignType(InventoriCampaignType campaignType) {
            this.campaignType = campaignType;
            return this;
        }

        public Builder withTargets(List<Target> targets) {
            this.targets = targets;
            return this;
        }

        public Builder withParameters(CampaignParameters parameters) {
            this.parameters = parameters;
            return this;
        }

        public Builder withIsSocial(@Nullable Boolean isSocial) {
            this.isSocial = isSocial;
            return this;
        }

        public Builder containsKeywordGroup(@Nullable Boolean containsKeywordGroup) {
            this.containsKeywordGroup = containsKeywordGroup;
            return this;
        }

        public CampaignPredictionRequest build() {
            return new CampaignPredictionRequest(campaignId, defaultTargetId, campaignType, targets, parameters,
                    containsKeywordGroup, isSocial, hasBrandlift);
        }
    }
}
