package ru.yandex.direct.inventori.model.request;

import java.util.List;
import java.util.Objects;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import org.jetbrains.annotations.NotNull;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.utils.CollectionUtils.defaultIfEmpty;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@JsonInclude(JsonInclude.Include.NON_NULL)
public class Target {

    @JsonProperty(value = "id")
    private Long adGroupId;

    @NotNull
    @JsonProperty(value = "group_type")
    private GroupType groupType;

    @JsonProperty(value = "banner_creatives")
    private List<BlockSize> blockSizes;

    @JsonProperty(value = "video_creatives")
    private List<VideoCreative> videoCreatives;

    @JsonProperty(value = "audio_creatives")
    private List<AudioCreative> audioCreatives;

    @JsonProperty(value = "genres_and_categories")
    private List<String> genresAndCategories;

    @JsonProperty(value = "excluded_bs_categories")
    private List<String> excludedBsCategories;

    @JsonProperty(value = "domains")
    private Set<String> domains;

    @JsonProperty(value = "excluded_domains")
    private Set<String> excludedDomains;

    @JsonProperty(value = "ssp")
    private Set<String> ssp;

    @JsonProperty(value = "excluded_ssp")
    private Set<String> excludedSsp;

    @JsonProperty(value = "crypta")
    private List<CryptaGroup> cryptaGroups;

    @JsonProperty(value = "regions")
    private Set<Integer> regions;

    @JsonProperty(value = "audiences")
    private List<AudienceGroup> audienceGroups;

    @JsonProperty(value = "platform_corrections")
    private PlatformCorrections platformCorrections;

    @JsonProperty(value = "profile_corrections")
    private List<ProfileCorrection> profileCorrections;

    @JsonInclude(value = JsonInclude.Include.CUSTOM, valueFilter = CampaignParametersCorrectionsFilter.class)
    @JsonProperty(value = "campaign_corrections")
    private CampaignParametersCorrections corrections;

    @JsonProperty(value = "page_blocks")
    private List<PageBlock> pageBlocks;

    @JsonProperty(value = "excluded_page_blocks")
    private List<PageBlock> excludedPageBlocks;

    @JsonIgnore
    private MainPageTrafficType mainPageTrafficType;

    @JsonProperty(value = "target_tags")
    private List<String> targetTags;

    @JsonProperty(value = "order_tags")
    private List<String> orderTags;

    @JsonProperty(value = "enable_nonskippable_video")
    private Boolean enableNonSkippableVideo;

    @JsonProperty(value = "is_default_group")
    private Boolean isDefaultGroup;

    @JsonProperty(value = "project_parameters")
    private List<ProjectParameter> projectParameters;

    /**
     * список форматов баннера
     */
    public List<BlockSize> getBlockSizes() {
        return blockSizes;
    }

    /**
     * список форматов видео-креативов и их длительностей
     */
    public List<VideoCreative> getVideoCreatives() {
        return videoCreatives;
    }

    /**
     * список минус-площадок
     */
    public Set<String> getExcludedDomains() {
        return excludedDomains;
    }

    /**
     * список разрешенных SSP-площадок
     */
    public Set<String> getSsp() {
        return ssp;
    }

    /**
     * список запрещенных SSP-площадок
     */
    public Set<String> getExcludedSsp() {
        return excludedSsp;
    }

    /**
     * группы (категории) сегментов крипты, расширенного соц. дем., интересов
     */
    public List<CryptaGroup> getCryptaGroups() {
        return cryptaGroups;
    }

    /**
     * метрика и я.аудитории
     */
    public List<AudienceGroup> getAudienceGroups() {
        return audienceGroups;
    }

    /**
     * список id регионов
     */
    public Set<Integer> getRegions() {
        return regions;
    }

    /**
     * корректировка по устройствам
     */
    public PlatformCorrections getPlatformCorrections() {
        return platformCorrections;
    }

    /**
     * корректировки на профиль пользователя
     */
    public List<ProfileCorrection> getProfileCorrections() {
        return profileCorrections;
    }

    /**
     * тип нацеливания
     */
    @NotNull
    public GroupType getGroupType() {
        return groupType;
    }

    /**
     * список форматов аудио-креативов и их длительностей
     */
    public List<AudioCreative> getAudioCreatives() {
        return audioCreatives;
    }

    public List<String> getGenresAndCategories() {
        return genresAndCategories;
    }

    public List<String> getExcludedBsCategories() {
        return excludedBsCategories;
    }

    /**
     * тип главной (только для таргетов с типом нацеливания MAIN_PAGE_AND_NTP)
     */
    public MainPageTrafficType getMainPageTrafficType() {
        return mainPageTrafficType;
    }

    public Long getAdGroupId() {
        return adGroupId;
    }

    public CampaignParametersCorrections getCorrections() {
        return corrections;
    }

    public List<PageBlock> getPageBlocks() {
        return pageBlocks;
    }

    public List<PageBlock> getExcludedPageBlocks() {
        return excludedPageBlocks;
    }

    public List<String> getTargetTags() {
        return targetTags;
    }

    public List<String> getOrderTags() {
        return orderTags;
    }

    public Boolean getEnableNonSkippableVideo() {
        return enableNonSkippableVideo;
    }

    public Boolean getIsDefaultGroup() {
        return isDefaultGroup;
    }

    public Set<String> getDomains() {
        return domains;
    }

    public List<ProjectParameter> getProjectParameters() {
        return projectParameters;
    }

    public Target withAdGroupId(Long adGroupId) {
        this.adGroupId = adGroupId;
        return this;
    }

    public Target withGroupType(@NotNull GroupType groupType) {
        this.groupType = groupType;
        return this;
    }

    public Target withBlockSizes(List<BlockSize> blockSizes) {
        this.blockSizes = defaultIfEmpty(blockSizes, null);
        return this;
    }

    public Target withVideoCreatives(List<VideoCreative> videoCreatives) {
        this.videoCreatives = defaultIfEmpty(videoCreatives, null);
        return this;
    }

    public Target withAudioCreatives(List<AudioCreative> audioCreatives) {
        this.audioCreatives = defaultIfEmpty(audioCreatives, null);
        return this;
    }

    public Target withGenresAndCategories(List<String> genres) {
        this.genresAndCategories = defaultIfEmpty(genres, null);
        return this;
    }

    public Target withExcludedBsCategories(List<String> excludedBsCategories) {
        this.excludedBsCategories = defaultIfEmpty(excludedBsCategories, null);
        return this;
    }

    public Target withExcludedDomains(Set<String> excludedDomains) {
        this.excludedDomains = defaultIfEmpty(excludedDomains, null);
        return this;
    }

    public Target withSsp(Set<String> ssp) {
        this.ssp = defaultIfEmpty(ssp, null);
        return this;
    }

    public Target withExcludedSsp(Set<String> excludedSsp) {
        this.excludedSsp = defaultIfEmpty(excludedSsp, null);
        return this;
    }

    public Target withCryptaGroups(List<CryptaGroup> cryptaGroups) {
        this.cryptaGroups = defaultIfEmpty(cryptaGroups, null);
        return this;
    }

    public Target withRegions(Set<Integer> regions) {
        this.regions = defaultIfEmpty(regions, null);
        return this;
    }

    public Target withAudienceGroups(List<AudienceGroup> audienceGroups) {
        this.audienceGroups = defaultIfEmpty(audienceGroups, null);
        return this;
    }

    public Target withPlatformCorrections(PlatformCorrections platformCorrections) {
        this.platformCorrections = platformCorrections;
        return this;
    }

    public Target withProfileCorrections(List<ProfileCorrection> profileCorrections) {
        this.profileCorrections = profileCorrections;
        return this;
    }

    public Target withCorrections(CampaignParametersCorrections corrections) {
        this.corrections = corrections;
        return this;
    }

    public Target withPageBlocks(List<PageBlock> pageBlocks) {
        this.pageBlocks = defaultIfEmpty(pageBlocks, null);
        return this;
    }

    public Target withExcludedPageBlocks(List<PageBlock> excludedPageBlocks) {
        this.excludedPageBlocks = defaultIfEmpty(excludedPageBlocks, null);
        return this;
    }

    public Target withMainPageTrafficType(MainPageTrafficType mainPageTrafficType) {
        this.mainPageTrafficType = mainPageTrafficType;
        return this;
    }

    public Target withTargetTags(List<String> targetTags) {
        this.targetTags = targetTags;
        return this;
    }

    public Target withOrderTags(List<String> orderTags) {
        this.orderTags = orderTags;
        return this;
    }

    public Target withEnableNonSkippableVideo(Boolean enableNonSkippableVideo) {
        this.enableNonSkippableVideo = enableNonSkippableVideo;
        return this;
    }

    public Target withIsDefaultGroup(Boolean isDefaultGroup) {
        this.isDefaultGroup = isDefaultGroup;
        return this;
    }

    public Target withDomains(Set<String> domains) {
        this.domains = defaultIfEmpty(domains, null);
        return this;
    }

    public Target withProjectParameters(List<ProjectParameter> projectParameters) {
        this.projectParameters = projectParameters;
        return this;
    }

    public boolean hasCreatives() {
        return !(nvl(this.getBlockSizes(), emptyList()).isEmpty() &&
                nvl(this.getVideoCreatives(), emptyList()).isEmpty() &&
                nvl(this.getAudioCreatives(), emptyList()).isEmpty());
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        Target target = (Target) o;
        return  Objects.equals(blockSizes, target.blockSizes) &&
                Objects.equals(videoCreatives, target.videoCreatives) &&
                Objects.equals(audioCreatives, target.audioCreatives) &&
                Objects.equals(genresAndCategories, target.genresAndCategories) &&
                Objects.equals(excludedBsCategories, target.excludedBsCategories) &&
                Objects.equals(excludedDomains, target.excludedDomains) &&
                Objects.equals(ssp, target.ssp) &&
                Objects.equals(excludedSsp, target.excludedSsp) &&
                Objects.equals(cryptaGroups, target.cryptaGroups) &&
                Objects.equals(regions, target.regions) &&
                Objects.equals(audienceGroups, target.audienceGroups) &&
                Objects.equals(platformCorrections, target.platformCorrections) &&
                Objects.equals(profileCorrections, target.profileCorrections) &&
                Objects.equals(corrections, target.corrections) &&
                groupType == target.groupType &&
                Objects.equals(pageBlocks, target.pageBlocks) &&
                Objects.equals(excludedPageBlocks, target.excludedPageBlocks) &&
                mainPageTrafficType == target.mainPageTrafficType &&
                Objects.equals(targetTags, target.targetTags) &&
                Objects.equals(enableNonSkippableVideo, target.enableNonSkippableVideo) &&
                Objects.equals(isDefaultGroup, target.isDefaultGroup) &&
                Objects.equals(domains, target.domains) &
                Objects.equals(projectParameters, target.projectParameters);
    }

    @Override
    public int hashCode() {
        return Objects.hash(blockSizes, videoCreatives, audioCreatives, genresAndCategories, excludedBsCategories,
                excludedDomains, cryptaGroups, regions, audienceGroups, platformCorrections, profileCorrections,
                corrections, groupType, pageBlocks, excludedPageBlocks, ssp, excludedSsp,
                mainPageTrafficType, targetTags, enableNonSkippableVideo, isDefaultGroup, domains, projectParameters);
    }

    @Override
    public String toString() {
        return "Target{" +
                "adGroupId=" + adGroupId +
                ", blockSizes=" + blockSizes +
                ", videoCreatives=" + videoCreatives +
                ", audioCreatives=" + audioCreatives +
                ", genresAndCategories=" + genresAndCategories +
                ", excludedBsCategories=" + excludedBsCategories +
                ", domains=" + domains +
                ", excludedDomains=" + excludedDomains +
                ", ssp=" + ssp +
                ", excludedSsp=" + excludedSsp +
                ", cryptaGroups=" + cryptaGroups +
                ", regions=" + regions +
                ", audienceGroups=" + audienceGroups +
                ", platformCorrections=" + platformCorrections +
                ", profileCorrections=" + profileCorrections +
                ", corrections=" + corrections +
                ", groupType=" + groupType +
                ", pageBlocks=" + pageBlocks +
                ", excludedPageBlocks=" + excludedPageBlocks +
                ", mainPageTrafficType=" + mainPageTrafficType +
                ", targetTags=" + targetTags +
                ", enableNonSkippableVideo=" + enableNonSkippableVideo +
                ", isDefaultGroup=" + isDefaultGroup +
                ", projectParameters=" + projectParameters +
                '}';
    }
}
