package ru.yandex.direct.jooqmapper;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Table;
import org.jooq.TableField;

import ru.yandex.direct.jooqmapper.commonread.CommonReader;
import ru.yandex.direct.jooqmapper.commonread.CommonReaderBuilder;
import ru.yandex.direct.jooqmapper.commonwrite.CommonWriter;
import ru.yandex.direct.jooqmapper.commonwrite.CommonWriterBuilder;
import ru.yandex.direct.jooqmapper.read.JooqReader;
import ru.yandex.direct.jooqmapper.write.JooqWriter;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;

import static java.util.Objects.requireNonNull;

/**
 * Маппер для чтения моделей из БД и записи моделей в БД.
 * Умеет читать из произвольных {@link Field}, в том числе вычисляемых.
 * Умеет по списку свойств модели отдавать список полей БД,
 * которые необходимо выбрать для их заполнения (см. {@link #getFieldsToRead(Collection)}).
 * <p>
 * Для создания используйте {@link JooqMapperBuilder}.
 *
 * @param <M> тип модели.
 */
// implements - для обратной совместимости с InsertHelper
@ParametersAreNonnullByDefault
public class JooqMapper<M extends Model> implements JooqModelToDbMapper<M>, JooqModelToDbFieldValuesMapper<M> {

    protected final CommonReader<M> commonReader;
    protected final CommonWriter<M> commonWriter;

    public JooqMapper(CommonReader<M> commonReader,
                      CommonWriter<M> commonWriter) {
        this.commonReader = requireNonNull(commonReader, "common jooq reader is required");
        this.commonWriter = requireNonNull(commonWriter, "common jooq writer is required");
    }

    public JooqMapper(JooqReader<M> jooqReader, JooqWriter<M> jooqWriter) {
        this(CommonReaderBuilder.builder(jooqReader).build(), CommonWriterBuilder.builder(jooqWriter).build());
    }

    /**
     * @see JooqReader#fromDb(Record, Model)
     */
    public M fromDb(Record record, M model) {
        return commonReader.fromDb(record, model);
    }

    /**
     * @see JooqReader#fromDb(Record, Model)
     */
    public M fromDb(Record record, M model, Set<Field> availableFields) {
        return commonReader.fromDb(record, model);
    }

    /**
     * @see JooqReader#fromDb(Record, Model, List)
     */
    public M fromDb(Record record, M model, List<ModelProperty<? super M, ?>> modelProperties) {
        return commonReader.fromDb(record, model, modelProperties);
    }

    /**
     * @see JooqReader#getFieldsToRead(Collection)
     */
    public Set<Field<?>> getFieldsToRead(Collection<ModelProperty<?, ?>> modelProperties) {
        return commonReader.getFieldsToRead(modelProperties);
    }

    /**
     * @see JooqReader#getFieldsToRead()
     */
    public Set<Field<?>> getFieldsToRead() {
        return commonReader.getFieldsToRead();
    }

    /**
     * @see JooqReader#canReadAtLeastOneProperty(Field[])
     */
    public boolean canReadAtLeastOneProperty(Field<?>[] fieldsCandidates) {
        return commonReader.canReadAtLeastOneProperty(fieldsCandidates);
    }

    /**
     * @see JooqReader#canReadAtLeastOneProperty(Collection)
     */
    public boolean canReadAtLeastOneProperty(Collection<? super Field<?>> fieldsCandidates) {
        return commonReader.canReadAtLeastOneProperty(fieldsCandidates);
    }

    @Override
    public <R extends Record> Map<TableField<R, ?>, ?> getDbFieldValues(M model, Table<R> table) {
        return commonWriter.getDbFieldValues(model, table);
    }

    @Override
    public <R extends Record> Map<TableField<R, ?>, ?> getDbFieldValues(
            M model,
            Table<R> table,
            Set<ModelProperty<? super M, ?>> requestedProperties,
            Set<ModelProperty<? super M, ?>> explicitValueProperties) {
        return commonWriter.getDbFieldValues(model, table, requestedProperties, explicitValueProperties);
    }

    public Set<ModelProperty<? super M, ?>> getReadableModelProperties() {
        return commonReader.getReadableModelProperties();
    }

    public Set<ModelProperty<? super M, ?>> getWritableModelProperties() {
        return commonWriter.getWritableModelProperties();
    }
}
