package ru.yandex.direct.jooqmapper;

import java.util.List;

import org.jooq.Record;
import org.jooq.TableField;

import ru.yandex.direct.jooqmapper.commonread.CommonReaderBuilder;
import ru.yandex.direct.jooqmapper.commonwrite.CommonWriterBuilder;
import ru.yandex.direct.jooqmapper.jsonread.JsonReader1Builder;
import ru.yandex.direct.jooqmapper.jsonread.JsonReaderBuilders;
import ru.yandex.direct.jooqmapper.jsonwrite.JsonWriter1Builder;
import ru.yandex.direct.jooqmapper.jsonwrite.JsonWriterBuilders;
import ru.yandex.direct.jooqmapper.read.DefaultReaderBuilder;
import ru.yandex.direct.jooqmapper.read.JooqReaderBuilder;
import ru.yandex.direct.jooqmapper.read.Reader1Builder;
import ru.yandex.direct.jooqmapper.read.Reader2Builder;
import ru.yandex.direct.jooqmapper.read.Reader3Builder;
import ru.yandex.direct.jooqmapper.read.ReaderBuilders;
import ru.yandex.direct.jooqmapper.write.DefaultWriterBuilder;
import ru.yandex.direct.jooqmapper.write.FunctionWriterBuilder;
import ru.yandex.direct.jooqmapper.write.JooqWriterBuilder;
import ru.yandex.direct.jooqmapper.write.SupplierWriterBuilder;
import ru.yandex.direct.jooqmapper.write.Writer1Builder;
import ru.yandex.direct.jooqmapper.write.Writer2Builder;
import ru.yandex.direct.jooqmapper.write.Writer3Builder;
import ru.yandex.direct.jooqmapper.write.WriterBuilders;
import ru.yandex.direct.jooqmapper.write.WriterToFieldBuilder;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Билдер для создания {@link JooqMapper}. Используйте вместе с {@link ReaderBuilders},
 * {@link WriterBuilders} и {@link ReaderWriterBuilders}.
 */
public class JooqMapperBuilder<M extends Model> {

    protected final CommonReaderBuilder<M> readerBuilder;
    protected final CommonWriterBuilder<M> writerBuilder;

    protected JooqMapperBuilder() {
        this(CommonReaderBuilder.builder(), CommonWriterBuilder.builder());
    }

    protected JooqMapperBuilder(JooqReaderBuilder<M> readerBuilder, JooqWriterBuilder<M> writerBuilder) {
        this(CommonReaderBuilder.builder(readerBuilder), CommonWriterBuilder.builder(writerBuilder));
    }

    protected JooqMapperBuilder(CommonReaderBuilder<M> readerBuilder,
                                CommonWriterBuilder<M> writerBuilder) {
        this.readerBuilder = readerBuilder;
        this.writerBuilder = writerBuilder;
    }

    public static <M extends Model> JooqMapperBuilder<M> builder() {
        return new JooqMapperBuilder<>();
    }

    public static <M extends Model> JooqMapperBuilder<M> builder(JooqMapper<M> sourceMapper) {
        return new JooqMapperBuilder<>(
                CommonReaderBuilder.builder(sourceMapper.commonReader),
                CommonWriterBuilder.builder(sourceMapper.commonWriter));
    }

    public static <M extends Model> JooqMapperBuilder<M> builder(
            List<? extends JooqMapper<? super M>> sourceMappers){

        var readers = mapList(sourceMappers, x -> x.commonReader);
        var writers = mapList(sourceMappers, x -> x.commonWriter);
        return new JooqMapperBuilder<>(
                CommonReaderBuilder.builder(readers),
                CommonWriterBuilder.builder(writers));
    }

    public <T, X extends Record, R> JooqMapperBuilder<M> map(
            ReaderWriterBuilder<M, T, X, R> readerWriterBuilder) {
        readerBuilder.readProperty(readerWriterBuilder.getProperty(),
                ReaderBuilders.fromField(readerWriterBuilder.getField())
                        .by(readerWriterBuilder.getReaderFunction()));
        writerBuilder.writeField(readerWriterBuilder.getField(),
                WriterBuilders.fromProperty(readerWriterBuilder.getProperty())
                        .by(readerWriterBuilder.getWriterFunction()));
        return this;
    }

    public <T, R> JooqMapperBuilder<M> readProperty(
            ModelProperty<? super M, T> modelProperty,
            Reader1Builder<R, T> reader1Builder
    ) {
        readerBuilder.readProperty(modelProperty, reader1Builder);
        return this;
    }

    public <T> JooqMapperBuilder<M> readProperty(
            ModelProperty<? super M, T> modelProperty,
            Reader1Builder.Reader1WithFieldStep<T> reader1Builder
    ) {
        readerBuilder.readProperty(modelProperty, reader1Builder);
        return this;
    }

    public <T, R1, R2> JooqMapperBuilder<M> readProperty(
            ModelProperty<? super M, T> modelProperty,
            Reader2Builder<R1, R2, T> reader2Builder
    ) {
        readerBuilder.readProperty(modelProperty, reader2Builder);
        return this;
    }

    public <T, R1, R2, R3> JooqMapperBuilder<M> readProperty(
            ModelProperty<? super M, T> modelProperty,
            Reader3Builder<R1, R2, R3, T> reader3Builder
    ) {
        readerBuilder.readProperty(modelProperty, reader3Builder);
        return this;
    }

    public <T> JooqMapperBuilder<M> readProperty(
            ModelProperty<? super M, T> modelProperty,
            DefaultReaderBuilder<T> readerBuilder
    ) {
        this.readerBuilder.readProperty(modelProperty, readerBuilder);
        return this;
    }

    public <X extends Record, T, R> JooqMapperBuilder<M> writeField(
            TableField<X, R> tableField,
            Writer1Builder<? super M, T, R> writer1Builder
    ) {
        writerBuilder.writeField(tableField, writer1Builder);
        return this;
    }

    /**
     * Позволяет избежать присваиваний поля самому себе вроде такого
     * {@code camp_options.strategy = camp_options.strategy}. Такие присваивания приводят к exception, если в enum
     * записана пустая строка и sql-mode строгий. Вместо этого будет явно указано значение из базы.
     */
    public <X extends Record, T, R> JooqMapperBuilder<M> writeFieldExplicitly(
            TableField<X, R> tableField,
            Writer1Builder<? super M, T, R> writer1Builder
    ) {
        writerBuilder.writeFieldExplicitly(tableField, writer1Builder);
        return this;
    }
    public <X extends Record, T, R> JooqMapperBuilder<M> writeField(
            TableField<X, R> tableField,
            WriterToFieldBuilder<? super M, T, R> writerToFieldBuilder
    ) {
        writerBuilder.writeField(tableField, writerToFieldBuilder);
        return this;
    }

    public <T> JooqMapperBuilder<M> writeField(
            TableField<?, T> tableField,
            Writer1Builder.Writer1WithPropertyStep<M, T> writer1Builder
    ) {
        writerBuilder.writeField(tableField, writer1Builder);
        return this;
    }

    public <X extends Record, T1, T2, R> JooqMapperBuilder<M> writeField(
            TableField<X, R> tableField,
            Writer2Builder<? super M, T1, T2, R> writer2Builder
    ) {
        writerBuilder.writeField(tableField, writer2Builder);
        return this;
    }

    public <X extends Record, T1, T2, T3, R> JooqMapperBuilder<M> writeField(
            TableField<X, R> tableField,
            Writer3Builder<? super M, T1, T2, T3, R> writer3Builder
    ) {
        writerBuilder.writeField(tableField, writer3Builder);
        return this;
    }

    public <X extends Record, R> JooqMapperBuilder<M> writeField(
            TableField<X, R> tableField,
            DefaultWriterBuilder<? super M, R> defaultWriterBuilder
    ) {
        writerBuilder.writeField(tableField, defaultWriterBuilder);
        return this;
    }

    public <X extends Record, R> JooqMapperBuilder<M> writeField(
            TableField<X, R> tableField,
            SupplierWriterBuilder<? super M, R> supplierWriterBuilder
    ) {
        writerBuilder.writeField(tableField, supplierWriterBuilder);
        return this;
    }

    public <X extends Record, R> JooqMapperBuilder<M> writeField(
            TableField<X, R> tableField,
            FunctionWriterBuilder<? super M, R> functionWriterBuilder
    ) {
        writerBuilder.writeField(tableField, functionWriterBuilder);
        return this;
    }

    // Работа с Json полями
    public <T, X extends Record> JooqMapperBuilder<M> map(
            JsonReaderWriterBuilder<M, T, X> readerWriterBuilder) {
        readerBuilder.readJsonProperty(readerWriterBuilder.getProperty(),
                JsonReaderBuilders.fromFieldAndPath(readerWriterBuilder.getField(), readerWriterBuilder.getJsonPath())
                        .by(readerWriterBuilder.getReaderFunction()));
        writerBuilder.writeField(readerWriterBuilder.getField(),
                JsonWriterBuilders.fromProperty(readerWriterBuilder.getProperty(), readerWriterBuilder.getJsonPath())
                        .by(readerWriterBuilder.getWriterFunction()));
        return this;
    }

    public <T> JooqMapperBuilder<M> readProperty(
            ModelProperty<? super M, T> modelProperty,
            JsonReader1Builder<T> reader1Builder
    ) {
        readerBuilder.readJsonProperty(modelProperty, reader1Builder);
        return this;
    }

    public <X extends Record, T> JooqMapperBuilder<M> writeField(
            TableField<X, String> tableField,
            JsonWriter1Builder<? super M, T> writer1Builder
    ) {
        writerBuilder.writeField(tableField, writer1Builder);
        return this;
    }

    public JooqMapper<M> build() {
        return new JooqMapper<>(readerBuilder.build(), writerBuilder.build());
    }
}
