package ru.yandex.direct.jooqmapper;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import org.jooq.CaseWhenStep;
import org.jooq.Condition;
import org.jooq.Field;
import org.jooq.QueryPart;
import org.jooq.Record;
import org.jooq.TableField;
import org.jooq.impl.DSL;

public class JooqMapperUtils {
    /**
     * Дефолтное значение количества записей при разбиении запроса на части
     */
    public static final int MINIMAL_PLAIN_CASE_SIZE = 10;

    private JooqMapperUtils() {
    }


    /**
     * Сгенерировать простое CASE-условие, в котором каждому значению будет соответствовать какой-то результат
     *
     * @param idField   - поле, содержащее id (по которому будет делаться проверка)
     * @param dataField - поле, которое надо обновить
     * @param values    - словарь id -> значение
     * @param <I>       - класс id
     * @param <V>       - класс значения
     * @return - sql, описывающий case
     */
    @Nonnull
    public static <I, V> Field<V> makeSimpleCaseStatement(Field<I> idField, Field<V> dataField, Map<I, V> values) {
        Iterator<I> iterator = values.keySet().iterator();

        if (!iterator.hasNext()) {
            return dataField;
        }

        // jooq использует разные типы для первого и последующих шагов. Поэтому используем итератор и while-loop
        I key = iterator.next();
        CaseWhenStep<I, V> caseWhenStep = DSL.choose(idField).when(key, values.get(key));

        while (iterator.hasNext()) {
            key = iterator.next();
            caseWhenStep = caseWhenStep.when(key, values.get(key));
        }

        return caseWhenStep.otherwise(dataField);
    }

    /**
     * Сгенерировать оптимизированное CASE-условие:
     * если ключей больше, чем MINIMAL_PLAIN_CASE_SIZE=10 и все ключи - числа, то строится бинарное дерево сравнений IF,
     * где в листьях небольшие CASE:
     * IF(f < 50, IF(f < 25, CASE..., CASE...), IF(f<75, CASE..., CASE...)))
     * <p>
     * MySQL выполняет case полным перебором, поэтому мы оптимизируем число сравнений
     *
     * @param idField   - поле, содержащее id (по которому будет делаться проверка)
     * @param dataField - поле, которое надо обновить
     * @param values    - словарь id -> значение
     * @param <R>       - класс jooq Record
     * @param <I>       - класс id
     * @param <V>       - класс значения
     * @return - sql, описывающий case
     */
    @Nonnull
    public static <R extends Record, I extends Number, V> Field<V> makeCaseStatement(
            TableField<R, I> idField,
            TableField<R, V> dataField,
            Map<I, V> values
    ) {
        return makeCaseStatementNoTable(idField, dataField, values);
    }

    /**
     * Аналог makeCaseStatement, но без привязки полей к таблице.
     */
    @Nonnull
    public static <I extends Number, V> Field<V> makeCaseStatementNoTable(
            Field<I> idField,
            Field<V> dataField,
            Map<I, V> values
    ) {
        if (values.size() <= MINIMAL_PLAIN_CASE_SIZE) {
            return makeSimpleCaseStatement(idField, dataField, values);
        } else {
            List<I> sortedIds = values.keySet().stream().sorted().collect(Collectors.toList());
            return makeBTreeCaseStatement(idField, dataField, sortedIds, values);
        }
    }

    @Nonnull
    private static <I extends Number, V> Field<V> makeBTreeCaseStatement(
            Field<I> idField,
            Field<V> dataField,
            List<I> sortedIds,
            Map<I, V> values
    ) {
        if (sortedIds.size() <= MINIMAL_PLAIN_CASE_SIZE) {
            // Collectors.toMap взрывается, если среди значений есть null
            Map<I, V> subValues = new HashMap<>();
            sortedIds.forEach(id -> subValues.put(id, values.get(id)));
            return makeSimpleCaseStatement(idField, dataField, subValues);
        }

        int index = sortedIds.size() / 2;
        I middle = sortedIds.get(index);

        return mysqlIf(idField.greaterOrEqual(middle),
                makeBTreeCaseStatement(idField, dataField, sortedIds.subList(index, sortedIds.size()), values),
                makeBTreeCaseStatement(idField, dataField, sortedIds.subList(0, index), values)
        );
    }

    /**
     * Генерит тернарный оператор - конструкцию вида
     * <pre>
     *     IF(condition, fieldIfTrue, fieldIfFalse)
     * </pre>
     *
     * @param condition    условие
     * @param fieldIfTrue  результат оператора, если условие истинно
     * @param fieldIfFalse результат оператора, если условие ложно
     * @see <a href="https://dev.mysql.com/doc/refman/5.7/en/control-flow-functions.html#function_if">If/else construct</a>
     */
    public static <T> Field<T> mysqlIf(Condition condition, Field<T> fieldIfTrue, Field<T> fieldIfFalse) {
        return mysqlIfInternal(condition, fieldIfTrue, fieldIfFalse);
    }

    /**
     * @see #mysqlIf(Condition, Field, Field)
     */
    public static <T> Field<T> mysqlIf(Field conditionField, Field<T> fieldIfTrue, Field<T> fieldIfFalse) {
        return mysqlIfInternal(conditionField, fieldIfTrue, fieldIfFalse);
    }

    private static <T> Field<T> mysqlIfInternal(QueryPart conditionalQueryPart, Field<T> fieldIfTrue,
                                                Field<T> fieldIfFalse) {
        return DSL.field("{if}({0}, {1}, {2})", fieldIfTrue.getDataType(), conditionalQueryPart, fieldIfTrue,
                fieldIfFalse);
    }
}
