package ru.yandex.direct.jooqmapper.commonread;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import one.util.streamex.StreamEx;
import org.jooq.Field;
import org.jooq.Record;

import ru.yandex.direct.jooqmapper.jsonread.JooqJsonReader;
import ru.yandex.direct.jooqmapper.read.JooqReader;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;

import static com.google.common.base.Preconditions.checkArgument;

@ParametersAreNonnullByDefault
public class CommonReader<M extends Model> {
    private final JooqReader<M> jooqReader;
    private final JooqJsonReader<M> jooqJsonReader;

    private final ImmutableSet<ModelProperty<? super M, ?>> readableModelProperties;

    private final ImmutableSet<ModelProperty<? super M, ?>> readableJsonModelProperties;

    public CommonReader(@Nullable JooqReader<M> jooqReader, @Nullable JooqJsonReader<M> jooqJsonReader) {
        checkArgument(jooqReader != null || jooqJsonReader != null, "readers is required");
        this.jooqReader = jooqReader;
        this.jooqJsonReader = jooqJsonReader;

        this.readableModelProperties = jooqReader != null ? jooqReader.getReadableModelProperties() : ImmutableSet.of();
        this.readableJsonModelProperties = jooqJsonReader != null ? jooqJsonReader.getReadableModelProperties() :
                ImmutableSet.of();
        checkArgument(!readableModelProperties.isEmpty() || !readableJsonModelProperties.isEmpty(),
                "readers map must not be empty");
        checkArgument(Sets.intersection(readableModelProperties, readableJsonModelProperties).isEmpty(),
                "ModelProperties intersects for different readers");
    }

    public M fromDb(Record record, M model) {
        if (jooqReader != null) {
            model = jooqReader.fromDb(record, model);
        }
        if (jooqJsonReader != null) {
            model = jooqJsonReader.fromDb(record, model);
        }
        return model;
    }

    public M fromDb(Record record, M model, Set<Field> availableFields) {
        if (jooqReader != null) {
            model = jooqReader.fromDb(record, model, availableFields);
        }
        if (jooqJsonReader != null) {
            model = jooqJsonReader.fromDb(record, model, availableFields);
        }
        return model;
    }

    public M fromDb(Record record, M model, List<ModelProperty<? super M, ?>> modelProperties) {
        // Выбираем ModelProperty, которые обрабатываются JooqJsonReader'ом
        List<ModelProperty<? super M, ?>> jsonModelProperties = StreamEx.of(modelProperties)
                .filter(readableJsonModelProperties::contains)
                .toList();
        // Убираем из переданных ModelProperty те, которые обрабатываются JooqJsonReader'ом
        List<ModelProperty<? super M, ?>> filteredModelProperties = StreamEx.of(modelProperties)
                .filter(Predicate.not(jsonModelProperties::contains))
                .toList();

        if (jooqReader != null) {
            model = jooqReader.fromDb(record, model, filteredModelProperties);
        }
        if (jooqJsonReader != null) {
            model = jooqJsonReader.fromDb(record, model, jsonModelProperties);
        }
        return model;
    }

    public Set<Field<?>> getFieldsToRead(Collection<ModelProperty<?, ?>> modelProperties) {
        Set<ModelProperty<?, ?>> jsonProperties = StreamEx.of(modelProperties)
                .filter(readableJsonModelProperties::contains)
                .toSet();

        Set<ModelProperty<?, ?>> properties = StreamEx.of(modelProperties)
                .filter(readableModelProperties::contains)
                .toSet();


        Set<Field<?>> fieldsToRead = new HashSet<>();
        if (jooqReader != null)
            fieldsToRead = jooqReader.getFieldsToRead(properties);

        if (jooqJsonReader != null) {
            fieldsToRead.addAll(jooqJsonReader.getFieldsToRead(jsonProperties));
        }

        return fieldsToRead;
    }

    public Set<Field<?>> getFieldsToRead() {
        Set<Field<?>> fieldsToRead = new HashSet<>();
        if (jooqReader != null) {
            fieldsToRead = jooqReader.getFieldsToRead();
        }
        if (jooqJsonReader != null) {
            fieldsToRead.addAll(jooqJsonReader.getFieldsToRead());
        }

        return fieldsToRead;
    }

    public boolean canReadAtLeastOneProperty(Field<?>[] fieldsCandidates) {
        return (jooqReader != null && jooqReader.canReadAtLeastOneProperty(fieldsCandidates)) ||
                (jooqJsonReader != null && jooqJsonReader.canReadAtLeastOneProperty(fieldsCandidates));
    }

    public boolean canReadAtLeastOneProperty(Collection<? super Field<?>> fieldsCandidates) {
        return (jooqReader != null && jooqReader.canReadAtLeastOneProperty(fieldsCandidates)) ||
                (jooqJsonReader != null && jooqJsonReader.canReadAtLeastOneProperty(fieldsCandidates));
    }

    public JooqReader<M> getJooqReader() {
        return jooqReader;
    }

    public JooqJsonReader<M> getJooqJsonReader() {
        return jooqJsonReader;
    }

    public Set<ModelProperty<? super M, ?>> getReadableModelProperties() {
        return Sets.union(readableModelProperties, readableJsonModelProperties);
    }
}
