package ru.yandex.direct.jooqmapper.commonwrite;

import java.util.Collections;
import java.util.Map;
import java.util.Set;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import one.util.streamex.StreamEx;
import org.jooq.Record;
import org.jooq.Table;
import org.jooq.TableField;

import ru.yandex.direct.jooqmapper.JooqModelToDbFieldValuesMapper;
import ru.yandex.direct.jooqmapper.JooqModelToDbMapper;
import ru.yandex.direct.jooqmapper.jsonwrite.JooqJsonWriter;
import ru.yandex.direct.jooqmapper.write.JooqWriter;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.utils.Collectors.nullFriendlyMapCollector;

@ParametersAreNonnullByDefault
public class CommonWriter<M extends Model> implements JooqModelToDbMapper<M>, JooqModelToDbFieldValuesMapper<M> {
    private final JooqWriter<M> jooqWriter;

    private final JooqJsonWriter<M> jooqJsonWriter;

    private final ImmutableSet<ModelProperty<? super M, ?>> writableModelProperties;

    private final ImmutableSet<ModelProperty<? super M, ?>> writableJsonModelProperties;

    public CommonWriter(@Nullable JooqWriter<M> jooqWriter, @Nullable JooqJsonWriter<M> jooqJsonWriter) {
        checkArgument(jooqWriter != null || jooqJsonWriter != null, "writers is required");

        this.jooqWriter = jooqWriter;
        this.jooqJsonWriter = jooqJsonWriter;

        this.writableModelProperties = jooqWriter != null
                ? jooqWriter.getWritableModelProperties() : ImmutableSet.of();

        this.writableJsonModelProperties = jooqJsonWriter != null
                ? jooqJsonWriter.getWritableModelProperties() : ImmutableSet.of();

        checkArgument(!writableModelProperties.isEmpty() || !writableJsonModelProperties.isEmpty(),
                "writers map must not be empty");
    }

    @Override
    public <R extends Record> Map<TableField<R, ?>, ?> getDbFieldValues(M model, Table<R> table) {
        Map<TableField<R, ?>, ?> dbFieldValues = jooqWriter != null ?
                jooqWriter.getDbFieldValues(model, table) :
                Collections.emptyMap();
        Map<TableField<R, ?>, ?> jsonDbFieldValues = jooqJsonWriter != null ?
                jooqJsonWriter.getDbFieldValues(model, table) :
                Collections.emptyMap();

        // Объединяем полученные значения полей
        // noinspection unchecked приведение типа безопасно, т.к. предварительно делается filter.
        return Stream.concat(
                dbFieldValues.entrySet().stream(),
                jsonDbFieldValues.entrySet().stream()
        ).collect(nullFriendlyMapCollector());
    }

    @Override
    public <R extends Record> Map<TableField<R, ?>, ?> getDbFieldValues(
            M model,
            Table<R> table,
            Set<ModelProperty<? super M, ?>> requestedProperties,
            Set<ModelProperty<? super M, ?>> explicitValueProperties) {
        // Разбираем переданные ModelProperty на обрабатываемые обычным Writer и JsonWriter
        Set<ModelProperty<? super M, ?>> properties = StreamEx.of(requestedProperties)
                .filter(writableModelProperties::contains)
                .toSet();

        Set<ModelProperty<? super M, ?>> jsonProperties = StreamEx.of(requestedProperties)
                .filter(writableJsonModelProperties::contains)
                .toSet();


        Map<TableField<R, ?>, ?> dbFieldValues = jooqWriter != null ?
                jooqWriter.getDbFieldValues(model, table, properties, explicitValueProperties) :
                Collections.emptyMap();
        Map<TableField<R, ?>, ?> jsonDbFieldValues = jooqJsonWriter != null ?
                jooqJsonWriter.getDbFieldValues(model, jsonProperties, table) :
                Collections.emptyMap();

        // Объединяем полученные значения полей
        // noinspection unchecked приведение типа безопасно, т.к. предварительно делается filter.
        return Stream.concat(
                dbFieldValues.entrySet().stream(),
                jsonDbFieldValues.entrySet().stream()
        ).collect(nullFriendlyMapCollector());
    }

    public JooqWriter<M> getJooqWriter() {
        return jooqWriter;
    }

    public JooqJsonWriter<M> getJooqJsonWriter() {
        return jooqJsonWriter;
    }

    public Set<ModelProperty<? super M, ?>> getWritableModelProperties() {
        return Sets.union(writableModelProperties, writableJsonModelProperties);
    }
}
